/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.server.blob.deduplication;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.UUID;
import nl.jqno.equalsverifier.EqualsVerifier;
import org.apache.commons.configuration2.Configuration;
import org.apache.commons.configuration2.PropertiesConfiguration;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.PlainBlobId;
import org.apache.james.server.blob.deduplication.GenerationAwareBlobId;
import org.apache.james.utils.UpdatableTickingClock;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.SoftAssertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

class GenerationAwareBlobIdTest {
    private static final Instant NOW = Instant.parse("2021-08-19T10:15:30.00Z");
    private BlobId.Factory delegate;
    private UpdatableTickingClock clock;
    private GenerationAwareBlobId.Factory testee;

    GenerationAwareBlobIdTest() {
    }

    @BeforeEach
    void setUp() {
        this.delegate = new PlainBlobId.Factory();
        this.clock = new UpdatableTickingClock(NOW);
        this.testee = new GenerationAwareBlobId.Factory((Clock)this.clock, this.delegate, GenerationAwareBlobId.Configuration.DEFAULT);
    }

    @Nested
    class ConfigurationPojo {
        ConfigurationPojo(GenerationAwareBlobIdTest this$0) {
        }

        @Test
        void shouldMatchPojoContract() {
            EqualsVerifier.forClass(GenerationAwareBlobId.Configuration.class).verify();
        }

        @Test
        void parseShouldReturnCorrectConfiguration() {
            PropertiesConfiguration configuration = new PropertiesConfiguration();
            configuration.addProperty("deduplication.gc.generation.duration", (Object)"15day");
            configuration.addProperty("deduplication.gc.generation.family", (Object)"2");
            Assertions.assertThat((Object)GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isEqualTo((Object)GenerationAwareBlobId.Configuration.builder().duration(Duration.ofDays(15L)).family(2));
        }

        @Test
        void defaultDurationUnitShouldBeDays() {
            PropertiesConfiguration configuration = new PropertiesConfiguration();
            configuration.addProperty("deduplication.gc.generation.duration", (Object)"15");
            configuration.addProperty("deduplication.gc.generation.family", (Object)"2");
            Assertions.assertThat((Object)GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isEqualTo((Object)GenerationAwareBlobId.Configuration.builder().duration(Duration.ofDays(15L)).family(2));
        }

        @Test
        void parseShouldReturnDefaultWhenNone() {
            PropertiesConfiguration configuration = new PropertiesConfiguration();
            Assertions.assertThat((Object)GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isEqualTo((Object)GenerationAwareBlobId.Configuration.DEFAULT);
        }

        @Nested
        class Failures {
            Failures(ConfigurationPojo this$1) {
            }

            @Test
            void shouldThrowOnZeroFamily() {
                PropertiesConfiguration configuration = new PropertiesConfiguration();
                configuration.addProperty("deduplication.gc.generation.duration", (Object)"15");
                configuration.addProperty("deduplication.gc.generation.family", (Object)"0");
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isInstanceOf(IllegalArgumentException.class);
            }

            @Test
            void shouldThrowOnNegativeFamily() {
                PropertiesConfiguration configuration = new PropertiesConfiguration();
                configuration.addProperty("deduplication.gc.generation.duration", (Object)"15");
                configuration.addProperty("deduplication.gc.generation.family", (Object)"-1");
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isInstanceOf(IllegalArgumentException.class);
            }

            @Test
            void shouldThrowOnZeroDuration() {
                PropertiesConfiguration configuration = new PropertiesConfiguration();
                configuration.addProperty("deduplication.gc.generation.duration", (Object)"0");
                configuration.addProperty("deduplication.gc.generation.family", (Object)"1");
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isInstanceOf(IllegalArgumentException.class);
            }

            @Test
            void shouldThrowOnNegativeDuration() {
                PropertiesConfiguration configuration = new PropertiesConfiguration();
                configuration.addProperty("deduplication.gc.generation.duration", (Object)"-5day");
                configuration.addProperty("deduplication.gc.generation.family", (Object)"1");
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobId.Configuration.parse((Configuration)configuration)).isInstanceOf(IllegalArgumentException.class);
            }
        }
    }

    @Nested
    class BlobIdPojo {
        BlobIdPojo() {
        }

        @Test
        void shouldMatchPojoContract() {
            EqualsVerifier.forClass(GenerationAwareBlobId.class).verify();
        }

        @Test
        void asStringShouldIntegrateFamilyAndGeneration() {
            GenerationAwareBlobId blobId = new GenerationAwareBlobId(23L, 456, GenerationAwareBlobIdTest.this.delegate.parse("abc"));
            Assertions.assertThat((String)blobId.asString()).isEqualTo("456_23_abc");
        }

        @Test
        void asStringShouldReturnDelegateForZeroFamily() {
            GenerationAwareBlobId blobId = new GenerationAwareBlobId(0L, 0, GenerationAwareBlobIdTest.this.delegate.parse("abc"));
            Assertions.assertThat((String)blobId.asString()).isEqualTo("abc");
        }

        @ParameterizedTest
        @ValueSource(strings={"1_2_abc", "abc"})
        void asStringShouldRevertFromString(String blobIdString) {
            GenerationAwareBlobId blobId = GenerationAwareBlobIdTest.this.testee.parse(blobIdString);
            Assertions.assertThat((String)blobId.asString()).isEqualTo(blobIdString);
        }

        @Test
        void noGenerationShouldNeverBeInActiveGeneration() {
            GenerationAwareBlobId blobId = new GenerationAwareBlobId(0L, 0, GenerationAwareBlobIdTest.this.delegate.parse("abc"));
            Assertions.assertThat((boolean)blobId.inActiveGeneration(GenerationAwareBlobId.Configuration.DEFAULT, NOW)).isFalse();
        }

        @Test
        void inActiveGenerationShouldReturnTrueWhenSameDate() {
            GenerationAwareBlobId blobId = GenerationAwareBlobIdTest.this.testee.of("abc");
            Assertions.assertThat((boolean)blobId.inActiveGeneration(GenerationAwareBlobId.Configuration.DEFAULT, NOW)).isTrue();
        }

        @Test
        void inActiveGenerationShouldReturnTrueWhenInTheFuture() {
            GenerationAwareBlobId blobId = GenerationAwareBlobIdTest.this.testee.of("abc");
            Assertions.assertThat((boolean)blobId.inActiveGeneration(GenerationAwareBlobId.Configuration.DEFAULT, NOW.minus(60L, ChronoUnit.DAYS))).isTrue();
        }

        @Test
        void inActiveGenerationShouldReturnTrueForAtLeastOneMoreMonth() {
            GenerationAwareBlobId blobId = GenerationAwareBlobIdTest.this.testee.of("abc");
            Assertions.assertThat((boolean)blobId.inActiveGeneration(GenerationAwareBlobId.Configuration.DEFAULT, NOW.plus(30L, ChronoUnit.DAYS))).isTrue();
        }

        @Test
        void inActiveGenerationShouldReturnFalseAfterTwoMonth() {
            GenerationAwareBlobId blobId = GenerationAwareBlobIdTest.this.testee.of("abc");
            Assertions.assertThat((boolean)blobId.inActiveGeneration(GenerationAwareBlobId.Configuration.DEFAULT, NOW.plus(60L, ChronoUnit.DAYS))).isFalse();
        }

        @Test
        void inActiveGenerationShouldReturnFalseWhenDistinctFamily() {
            GenerationAwareBlobId blobId = new GenerationAwareBlobId(628L, 2, GenerationAwareBlobIdTest.this.delegate.of("abcd"));
            Assertions.assertThat((boolean)blobId.inActiveGeneration(GenerationAwareBlobId.Configuration.DEFAULT, NOW.plus(60L, ChronoUnit.DAYS))).isFalse();
        }
    }

    @Nested
    class BlobIdParsing {
        BlobIdParsing() {
        }

        @Test
        void previousBlobIdsShouldBeParsable() {
            String blobIdString = GenerationAwareBlobIdTest.this.delegate.of("abc").asString();
            GenerationAwareBlobId actual = GenerationAwareBlobIdTest.this.testee.parse(blobIdString);
            SoftAssertions.assertSoftly(soft -> {
                soft.assertThat(actual.getFamily()).isEqualTo(0);
                soft.assertThat(actual.getGeneration()).isEqualTo(0L);
                soft.assertThat(actual.getDelegate().asString()).isEqualTo(blobIdString);
            });
        }

        @Test
        void noFamilyShouldBeParsable() {
            String blobIdString = "0_0_" + GenerationAwareBlobIdTest.this.delegate.of("abc").asString();
            GenerationAwareBlobId actual = GenerationAwareBlobIdTest.this.testee.parse(blobIdString);
            SoftAssertions.assertSoftly(soft -> {
                soft.assertThat(actual.getFamily()).isEqualTo(0);
                soft.assertThat(actual.getGeneration()).isEqualTo(0L);
                soft.assertThat((Object)actual.getDelegate()).isEqualTo((Object)GenerationAwareBlobIdTest.this.delegate.of("abc"));
            });
        }

        @Test
        void generationBlobIdShouldBeParsable() {
            String blobIdString = "12_126_" + GenerationAwareBlobIdTest.this.delegate.of("abc").asString();
            GenerationAwareBlobId actual = GenerationAwareBlobIdTest.this.testee.parse(blobIdString);
            SoftAssertions.assertSoftly(soft -> {
                soft.assertThat(actual.getFamily()).isEqualTo(12);
                soft.assertThat(actual.getGeneration()).isEqualTo(126L);
                soft.assertThat((Object)actual.getDelegate()).isEqualTo((Object)GenerationAwareBlobIdTest.this.delegate.of("abc"));
            });
        }

        @Test
        void wrappedBlobIdCanContainSeparator() {
            String blobIdString = "12_126_ab_c";
            GenerationAwareBlobId actual = GenerationAwareBlobIdTest.this.testee.parse(blobIdString);
            SoftAssertions.assertSoftly(soft -> {
                soft.assertThat(actual.getFamily()).isEqualTo(12);
                soft.assertThat(actual.getGeneration()).isEqualTo(126L);
                soft.assertThat((Object)actual.getDelegate()).isEqualTo((Object)GenerationAwareBlobIdTest.this.delegate.parse("ab_c"));
            });
        }

        @ParameterizedTest
        @ValueSource(strings={"abc", "abc_", "1_abc", "1_2", "1_2_", "_abc"})
        void fromShouldFallbackWhenNotApplicable(String blobIdString) {
            GenerationAwareBlobId actual = GenerationAwareBlobIdTest.this.testee.parse(blobIdString);
            SoftAssertions.assertSoftly(soft -> {
                soft.assertThat(actual.getFamily()).isEqualTo(0);
                soft.assertThat(actual.getGeneration()).isEqualTo(0L);
                soft.assertThat((Object)actual.getDelegate()).isEqualTo((Object)GenerationAwareBlobIdTest.this.delegate.parse(blobIdString));
            });
        }

        @Nested
        class Failures {
            Failures() {
            }

            @Test
            void emptyShouldFail() {
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobIdTest.this.testee.parse("")).isInstanceOf(IllegalArgumentException.class);
            }

            @Test
            void nullShouldFailShouldFail() {
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobIdTest.this.testee.parse(null)).isInstanceOf(NullPointerException.class);
            }

            @ParameterizedTest
            @ValueSource(strings={"invalid_2_abc", "1_invalid_abc", "1__abc", "__abc", "_1_abc", "-1_2_abc", "1_-1_abc"})
            void fromShouldFallbackWhenNotApplicable(String blobIdString) {
                Assertions.assertThatThrownBy(() -> GenerationAwareBlobIdTest.this.testee.parse(blobIdString)).isInstanceOf(IllegalArgumentException.class);
            }
        }
    }

    @Nested
    class BlobIdGeneration {
        BlobIdGeneration() {
        }

        @Test
        void ofShouldGenerateABlobIdOfTheRightGeneration() {
            String key = UUID.randomUUID().toString();
            GenerationAwareBlobId actual = GenerationAwareBlobIdTest.this.testee.of(key);
            SoftAssertions.assertSoftly(soft -> {
                soft.assertThat(actual.getFamily()).isEqualTo(GenerationAwareBlobId.Configuration.DEFAULT.getFamily());
                soft.assertThat(actual.getGeneration()).isEqualTo(628L);
                soft.assertThat((Object)actual.getDelegate()).isEqualTo((Object)GenerationAwareBlobIdTest.this.delegate.of(key));
            });
        }
    }
}

