/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.server.blob.deduplication;

import java.io.InputStream;
import java.time.Clock;
import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.BlobReferenceSource;
import org.apache.james.blob.api.BlobStore;
import org.apache.james.blob.api.BlobStoreDAO;
import org.apache.james.blob.api.BucketName;
import org.apache.james.blob.api.ObjectNotFoundException;
import org.apache.james.blob.api.PlainBlobId;
import org.apache.james.server.blob.deduplication.BloomFilterGCAlgorithm;
import org.apache.james.server.blob.deduplication.DeDuplicationBlobStore;
import org.apache.james.server.blob.deduplication.GenerationAwareBlobId;
import org.apache.james.task.Task;
import org.apache.james.utils.UpdatableTickingClock;
import org.assertj.core.api.AssertionsForClassTypes;
import org.awaitility.Awaitility;
import org.awaitility.Durations;
import org.awaitility.core.ConditionFactory;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.RepeatedTest;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public interface BloomFilterGCAlgorithmContract {
    public static final PlainBlobId.Factory BLOB_ID_FACTORY = new PlainBlobId.Factory();
    public static final ZonedDateTime NOW = ZonedDateTime.parse("2015-10-30T16:12:00Z");
    public static final BucketName DEFAULT_BUCKET = BucketName.of((String)"default");
    public static final GenerationAwareBlobId.Configuration GENERATION_AWARE_BLOB_ID_CONFIGURATION = GenerationAwareBlobId.Configuration.DEFAULT;
    public static final int EXPECTED_BLOB_COUNT = 100;
    public static final int DELETION_WINDOW_SIZE = 10;
    public static final double ASSOCIATED_PROBABILITY = 0.01;
    public static final ConditionFactory CALMLY_AWAIT = Awaitility.with().pollInterval(Durations.ONE_HUNDRED_MILLISECONDS).and().pollDelay(Durations.ONE_HUNDRED_MILLISECONDS).await().atMost(Durations.TEN_SECONDS);
    public static final BlobReferenceSource BLOB_REFERENCE_SOURCE = (BlobReferenceSource)Mockito.mock(BlobReferenceSource.class);
    public static final UpdatableTickingClock CLOCK = new UpdatableTickingClock(NOW.toInstant());
    public static final GenerationAwareBlobId.Factory GENERATION_AWARE_BLOB_ID_FACTORY = new GenerationAwareBlobId.Factory((Clock)CLOCK, (BlobId.Factory)BLOB_ID_FACTORY, GENERATION_AWARE_BLOB_ID_CONFIGURATION);

    public BlobStoreDAO blobStoreDAO();

    @BeforeEach
    default public void setUp() {
        CLOCK.setInstant(NOW.toInstant());
    }

    default public BlobStore blobStore() {
        return new DeDuplicationBlobStore(this.blobStoreDAO(), DEFAULT_BUCKET, (BlobId.Factory)GENERATION_AWARE_BLOB_ID_FACTORY);
    }

    default public BloomFilterGCAlgorithm bloomFilterGCAlgorithm() {
        return new BloomFilterGCAlgorithm(BLOB_REFERENCE_SOURCE, this.blobStoreDAO(), GENERATION_AWARE_BLOB_ID_FACTORY, GENERATION_AWARE_BLOB_ID_CONFIGURATION, (Clock)CLOCK);
    }

    @RepeatedTest(value=10)
    default public void gcShouldRemoveOrphanBlob() {
        BlobStore blobStore = this.blobStore();
        BlobId blobId = (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
        Mockito.when((Object)BLOB_REFERENCE_SOURCE.listReferencedBlobs()).thenReturn((Object)Flux.empty());
        CLOCK.setInstant(NOW.plusMonths(2L).toInstant());
        BloomFilterGCAlgorithm.Context context = new BloomFilterGCAlgorithm.Context(100L, 0.01);
        BloomFilterGCAlgorithm bloomFilterGCAlgorithm = this.bloomFilterGCAlgorithm();
        Task.Result result = (Task.Result)Mono.from((Publisher)bloomFilterGCAlgorithm.gc(100, 10, 0.01, DEFAULT_BUCKET, context)).block();
        AssertionsForClassTypes.assertThat((Object)result).isEqualTo((Object)Task.Result.COMPLETED);
        AssertionsForClassTypes.assertThat((Object)context.snapshot()).isEqualTo((Object)BloomFilterGCAlgorithm.Context.Snapshot.builder().referenceSourceCount(0L).blobCount(1L).gcedBlobCount(1L).errorCount(0L).bloomFilterExpectedBlobCount(100L).bloomFilterAssociatedProbability(0.01).build());
        AssertionsForClassTypes.assertThatThrownBy(() -> blobStore.read(DEFAULT_BUCKET, blobId)).isInstanceOf(ObjectNotFoundException.class);
    }

    @Test
    default public void gcShouldNotRemoveUnExpireBlob() {
        BlobStore blobStore = this.blobStore();
        BlobId blobId = (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
        Mockito.when((Object)BLOB_REFERENCE_SOURCE.listReferencedBlobs()).thenReturn((Object)Flux.empty());
        BloomFilterGCAlgorithm.Context context = new BloomFilterGCAlgorithm.Context(100L, 0.01);
        BloomFilterGCAlgorithm bloomFilterGCAlgorithm = this.bloomFilterGCAlgorithm();
        Task.Result result = (Task.Result)Mono.from((Publisher)bloomFilterGCAlgorithm.gc(100, 10, 0.01, DEFAULT_BUCKET, context)).block();
        AssertionsForClassTypes.assertThat((Object)result).isEqualTo((Object)Task.Result.COMPLETED);
        AssertionsForClassTypes.assertThat((Object)context.snapshot()).isEqualTo((Object)BloomFilterGCAlgorithm.Context.Snapshot.builder().referenceSourceCount(0L).blobCount(1L).gcedBlobCount(0L).errorCount(0L).bloomFilterExpectedBlobCount(100L).bloomFilterAssociatedProbability(0.01).build());
        AssertionsForClassTypes.assertThat((InputStream)blobStore.read(DEFAULT_BUCKET, blobId)).isNotNull();
    }

    @RepeatedTest(value=10)
    default public void gcShouldNotRemoveReferencedBlob() {
        BlobStore blobStore = this.blobStore();
        BlobId blobId = (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
        Mockito.when((Object)BLOB_REFERENCE_SOURCE.listReferencedBlobs()).thenReturn((Object)Flux.just((Object)blobId));
        BloomFilterGCAlgorithm.Context context = new BloomFilterGCAlgorithm.Context(100L, 0.01);
        BloomFilterGCAlgorithm bloomFilterGCAlgorithm = this.bloomFilterGCAlgorithm();
        Task.Result result = (Task.Result)Mono.from((Publisher)bloomFilterGCAlgorithm.gc(100, 10, 0.01, DEFAULT_BUCKET, context)).block();
        AssertionsForClassTypes.assertThat((Object)result).isEqualTo((Object)Task.Result.COMPLETED);
        AssertionsForClassTypes.assertThat((Object)context.snapshot()).isEqualTo((Object)BloomFilterGCAlgorithm.Context.Snapshot.builder().referenceSourceCount(1L).blobCount(1L).gcedBlobCount(0L).errorCount(0L).bloomFilterExpectedBlobCount(100L).bloomFilterAssociatedProbability(0.01).build());
        AssertionsForClassTypes.assertThat((InputStream)blobStore.read(DEFAULT_BUCKET, blobId)).isNotNull();
    }

    @Test
    default public void gcShouldSuccessWhenMixCase() {
        BlobStore blobStore = this.blobStore();
        List<BlobId> referencedBlobIds = IntStream.range(0, 100).mapToObj(index -> (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block()).collect(Collectors.toList());
        List orphanBlobIds = IntStream.range(0, 50).mapToObj(index -> (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block()).collect(Collectors.toList());
        Mockito.when((Object)BLOB_REFERENCE_SOURCE.listReferencedBlobs()).thenReturn((Object)Flux.fromIterable(referencedBlobIds));
        CLOCK.setInstant(NOW.plusMonths(2L).toInstant());
        BloomFilterGCAlgorithm.Context context = new BloomFilterGCAlgorithm.Context(100L, 0.01);
        BloomFilterGCAlgorithm bloomFilterGCAlgorithm = this.bloomFilterGCAlgorithm();
        Task.Result result = (Task.Result)Mono.from((Publisher)bloomFilterGCAlgorithm.gc(100, 10, 0.01, DEFAULT_BUCKET, context)).block();
        AssertionsForClassTypes.assertThat((Object)result).isEqualTo((Object)Task.Result.COMPLETED);
        BloomFilterGCAlgorithm.Context.Snapshot snapshot = context.snapshot();
        AssertionsForClassTypes.assertThat((long)snapshot.getReferenceSourceCount()).isEqualTo((long)referencedBlobIds.size());
        AssertionsForClassTypes.assertThat((long)snapshot.getBlobCount()).isEqualTo((long)(referencedBlobIds.size() + orphanBlobIds.size()));
        AssertionsForClassTypes.assertThat((long)snapshot.getGcedBlobCount()).isLessThanOrEqualTo((long)orphanBlobIds.size()).isGreaterThan(0L);
        referencedBlobIds.forEach(blobId -> AssertionsForClassTypes.assertThat((InputStream)blobStore.read(DEFAULT_BUCKET, blobId)).isNotNull());
    }

    @Test
    default public void allOrphanBlobIdsShouldRemovedAfterMultipleRunningTimesGC() {
        BlobStore blobStore = this.blobStore();
        List referencedBlobIds = IntStream.range(0, 100).mapToObj(index -> (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block()).collect(Collectors.toList());
        List orphanBlobIds = IntStream.range(0, 50).mapToObj(index -> (BlobId)Mono.from((Publisher)blobStore.save(DEFAULT_BUCKET, UUID.randomUUID().toString(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block()).collect(Collectors.toList());
        Mockito.when((Object)BLOB_REFERENCE_SOURCE.listReferencedBlobs()).thenReturn((Object)Flux.fromIterable(referencedBlobIds));
        CLOCK.setInstant(NOW.plusMonths(2L).toInstant());
        CALMLY_AWAIT.untilAsserted(() -> {
            Mono.from((Publisher)this.bloomFilterGCAlgorithm().gc(100, 10, 0.01, DEFAULT_BUCKET, new BloomFilterGCAlgorithm.Context(100L, 0.01))).block();
            orphanBlobIds.forEach(blobId -> AssertionsForClassTypes.assertThatThrownBy(() -> blobStore.read(DEFAULT_BUCKET, blobId)).isInstanceOf(ObjectNotFoundException.class));
        });
    }

    @Test
    default public void gcShouldHandlerErrorWhenException() {
        Mockito.when((Object)BLOB_REFERENCE_SOURCE.listReferencedBlobs()).thenReturn((Object)Flux.empty());
        BlobStoreDAO blobStoreDAO = (BlobStoreDAO)Mockito.mock(BlobStoreDAO.class);
        GenerationAwareBlobId blobId = GENERATION_AWARE_BLOB_ID_FACTORY.of(UUID.randomUUID().toString());
        Mockito.when((Object)blobStoreDAO.listBlobs(DEFAULT_BUCKET)).thenReturn((Object)Flux.just((Object)blobId));
        Mockito.when((Object)blobStoreDAO.delete((BucketName)ArgumentMatchers.eq((Object)DEFAULT_BUCKET), (Collection)ArgumentMatchers.any(Collection.class))).thenReturn((Object)Mono.error((Throwable)new RuntimeException("test")));
        CLOCK.setInstant(NOW.plusMonths(2L).toInstant());
        BloomFilterGCAlgorithm.Context context = new BloomFilterGCAlgorithm.Context(100L, 0.01);
        BloomFilterGCAlgorithm bloomFilterGCAlgorithm = new BloomFilterGCAlgorithm(BLOB_REFERENCE_SOURCE, blobStoreDAO, GENERATION_AWARE_BLOB_ID_FACTORY, GENERATION_AWARE_BLOB_ID_CONFIGURATION, (Clock)CLOCK);
        Task.Result result = (Task.Result)Mono.from((Publisher)bloomFilterGCAlgorithm.gc(100, 10, 0.01, DEFAULT_BUCKET, context)).block();
        AssertionsForClassTypes.assertThat((Object)result).isEqualTo((Object)Task.Result.PARTIAL);
        AssertionsForClassTypes.assertThat((Object)context.snapshot()).isEqualTo((Object)BloomFilterGCAlgorithm.Context.Snapshot.builder().referenceSourceCount(0L).blobCount(1L).gcedBlobCount(0L).errorCount(1L).bloomFilterExpectedBlobCount(100L).bloomFilterAssociatedProbability(0.01).build());
    }
}

