/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.blob.cassandra.cache;

import com.datastax.oss.driver.api.core.CqlSession;
import com.google.common.base.Strings;
import com.google.common.io.ByteSource;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import javax.mail.internet.MimeMessage;
import org.apache.james.backends.cassandra.CassandraCluster;
import org.apache.james.backends.cassandra.CassandraClusterExtension;
import org.apache.james.backends.cassandra.Scenario;
import org.apache.james.backends.cassandra.StatementRecorder;
import org.apache.james.backends.cassandra.components.CassandraModule;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.BlobStore;
import org.apache.james.blob.api.BlobStoreContract;
import org.apache.james.blob.api.BucketName;
import org.apache.james.blob.api.HashBlobId;
import org.apache.james.blob.api.ObjectNotFoundException;
import org.apache.james.blob.api.Store;
import org.apache.james.blob.api.TestBlobId;
import org.apache.james.blob.cassandra.CassandraBlobModule;
import org.apache.james.blob.cassandra.CassandraBlobStoreFactory;
import org.apache.james.blob.cassandra.cache.BlobStoreCache;
import org.apache.james.blob.cassandra.cache.BlobStoreCacheContract;
import org.apache.james.blob.cassandra.cache.CachedBlobStore;
import org.apache.james.blob.cassandra.cache.CassandraBlobCacheModule;
import org.apache.james.blob.cassandra.cache.CassandraBlobStoreCache;
import org.apache.james.blob.cassandra.cache.CassandraCacheConfiguration;
import org.apache.james.blob.mail.MimeMessagePartsId;
import org.apache.james.blob.mail.MimeMessageStore;
import org.apache.james.core.builder.MimeMessageBuilder;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.metrics.tests.RecordingMetricFactory;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.CollectionAssert;
import org.assertj.core.api.IntegerAssert;
import org.assertj.core.api.SoftAssertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Mono;

public class CachedBlobStoreTest
implements BlobStoreContract {
    private static final BucketName DEFAULT_BUCKETNAME = BucketName.DEFAULT;
    private static final BucketName TEST_BUCKETNAME = BucketName.of((String)"test");
    private static final byte[] APPROXIMATELY_FIVE_KILOBYTES = Strings.repeat((String)"0123456789\n", (int)500).getBytes(StandardCharsets.UTF_8);
    @RegisterExtension
    static CassandraClusterExtension cassandraCluster = new CassandraClusterExtension(CassandraModule.aggregateModules((CassandraModule[])new CassandraModule[]{CassandraBlobModule.MODULE, CassandraBlobCacheModule.MODULE}));
    private BlobStore testee;
    private BlobStore backend;
    private BlobStoreCache cache;
    private RecordingMetricFactory metricFactory;

    @BeforeEach
    void setUp(CassandraCluster cassandra) {
        this.backend = CassandraBlobStoreFactory.forTesting((CqlSession)cassandra.getConf(), (MetricFactory)new RecordingMetricFactory()).passthrough();
        CassandraCacheConfiguration cacheConfig = new CassandraCacheConfiguration.Builder().sizeThresholdInBytes(BlobStoreCacheContract.EIGHT_KILOBYTES.length + 1).build();
        this.metricFactory = new RecordingMetricFactory();
        this.cache = new CassandraBlobStoreCache((CqlSession)cassandra.getConf(), cacheConfig);
        this.testee = new CachedBlobStore(this.cache, this.backend, cacheConfig, (MetricFactory)this.metricFactory);
    }

    public BlobStore testee() {
        return this.testee;
    }

    public BlobId.Factory blobIdFactory() {
        return new HashBlobId.Factory();
    }

    @Test
    public void shouldCacheWhenDefaultBucketName() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, BlobStoreCacheContract.EIGHT_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        byte[] actual = (byte[])Mono.from((Publisher)this.cache.read(blobId)).block();
        Assertions.assertThat((byte[])actual).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
    }

    @Test
    public void shouldNotCacheWhenNotDefaultBucketName() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(TEST_BUCKETNAME, BlobStoreCacheContract.EIGHT_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(TEST_BUCKETNAME, blobId)).block()).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
        });
    }

    @Test
    public void shouldNotCacheWhenDefaultBucketNameAndBigByteDataAndSizeBase() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, TWELVE_MEGABYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(TWELVE_MEGABYTES);
        });
    }

    @Test
    public void shouldSavedBothInCacheAndBackendWhenSizeBase() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, BlobStoreCacheContract.EIGHT_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((byte[])Mono.from((Publisher)this.cache.read(blobId)).block()).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
        });
    }

    @Test
    public void shouldSavedBothInCacheAndBackendWhenHighPerformance() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, BlobStoreCacheContract.EIGHT_KILOBYTES, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((byte[])Mono.from((Publisher)this.cache.read(blobId)).block()).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
        });
    }

    @Test
    public void shouldNotCacheWhenLowCost() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, BlobStoreCacheContract.EIGHT_KILOBYTES, BlobStore.StoragePolicy.LOW_COST)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(BlobStoreCacheContract.EIGHT_KILOBYTES);
        });
    }

    @Test
    public void shouldCacheWhenEmptyStream() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, (InputStream)new ByteArrayInputStream(EMPTY_BYTEARRAY), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(EMPTY_BYTEARRAY));
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(EMPTY_BYTEARRAY);
        });
    }

    @Test
    public void shouldCacheWhenEmptyByteSource() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, ByteSource.wrap((byte[])EMPTY_BYTEARRAY), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(EMPTY_BYTEARRAY));
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(EMPTY_BYTEARRAY);
        });
    }

    @Test
    public void shouldNotCacheWhenEmptyByteArray() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, EMPTY_BYTEARRAY, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(EMPTY_BYTEARRAY));
            softly.assertThat((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).containsExactly(EMPTY_BYTEARRAY);
        });
    }

    @Test
    public void shouldCacheWhenFiveKilobytesSteam() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, (InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
        });
    }

    @Test
    public void shouldCacheWhenFiveKilobytesByteSource() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, ByteSource.wrap((byte[])APPROXIMATELY_FIVE_KILOBYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
        });
    }

    @Test
    public void shouldCacheWhenFiveKilobytesBytes() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
        });
    }

    @Test
    public void shouldRemoveBothInCacheAndBackendWhenDefaultBucketName() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee().save(DEFAULT_BUCKETNAME, BlobStoreCacheContract.EIGHT_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThatCode(() -> ((Mono)Mono.from((Publisher)this.testee().delete(DEFAULT_BUCKETNAME, blobId))).block()).doesNotThrowAnyException();
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThatThrownBy(() -> Mono.from((Publisher)this.backend.readBytes(DEFAULT_BUCKETNAME, blobId)).block()).isInstanceOf(ObjectNotFoundException.class);
        });
    }

    @Test
    public void shouldCacheWhenReadBytesWithDefaultBucket() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.testee().readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
        });
    }

    @Test
    public void shouldCacheWhenReadWithDefaultBucket() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat(this.testee().read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.cache.read(blobId)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
        });
    }

    @Test
    public void shouldNotCacheWhenReadBytesWithOutDefaultBucket() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(TEST_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.testee().readBytes(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
        });
    }

    @Test
    public void shouldNotCacheWhenReadWithOutDefaultBucket() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(TEST_BUCKETNAME, (InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat(this.testee().read(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
        });
    }

    @Test
    public void shouldNotCacheWhenReadWithBigByteArray() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(DEFAULT_BUCKETNAME, (InputStream)new ByteArrayInputStream(TWELVE_MEGABYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.testee().readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(TWELVE_MEGABYTES));
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
        });
    }

    @Test
    public void shouldNotCacheWhenReadWithBigStream() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee.save(DEFAULT_BUCKETNAME, (InputStream)new ByteArrayInputStream(TWELVE_MEGABYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat(this.testee().read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).hasSameContentAs((InputStream)new ByteArrayInputStream(TWELVE_MEGABYTES));
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
        });
    }

    @Test
    public void shouldNotCacheWhenReadWithOutDefaultBucketByteSource() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(TEST_BUCKETNAME, ByteSource.wrap((byte[])APPROXIMATELY_FIVE_KILOBYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat(this.testee().read(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
        });
    }

    @Test
    public void shouldNotCacheWhenReadWithBigByteSource() {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.backend.save(DEFAULT_BUCKETNAME, ByteSource.wrap((byte[])TWELVE_MEGABYTES), BlobStore.StoragePolicy.SIZE_BASED)).block();
        SoftAssertions.assertSoftly(softly -> {
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
            softly.assertThat((InputStream)new ByteArrayInputStream((byte[])Mono.from((Publisher)this.testee().readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block())).hasSameContentAs((InputStream)new ByteArrayInputStream(TWELVE_MEGABYTES));
            softly.assertThat(Mono.from((Publisher)this.cache.read(blobId)).blockOptional()).isEmpty();
        });
    }

    @Test
    public void readShouldNotPropagateFailures(CassandraCluster cassandra) {
        BlobId blobId = (BlobId)Mono.from((Publisher)this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
        cassandra.getConf().registerScenario(new Scenario.ExecutionHook[]{Scenario.Builder.fail().times(1).whenQueryStartsWith("SELECT * FROM blob_cache WHERE id=:id")});
        Mono.from((Publisher)this.cache.read(blobId)).block();
        Assertions.assertThat((InputStream)this.testee().read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).hasSameContentAs((InputStream)new ByteArrayInputStream(APPROXIMATELY_FIVE_KILOBYTES));
    }

    @Test
    public void cachedBlobStoreShouldOnlyBeQueriedForHeaders(CassandraCluster cassandra) throws Exception {
        cassandra.getConf().printStatements();
        MimeMessage message = MimeMessageBuilder.mimeMessageBuilder().addHeader("Date", "Thu, 6 Sep 2018 13:29:13 +0700 (ICT)").addHeader("Message-ID", "<84739718.0.1536215353507@localhost.localdomain>").addFrom("any@any.com").addToRecipient("toddy@any.com").setSubject("Important Mail").setText("Important mail content").build();
        Store mimeMessageStore = new MimeMessageStore.Factory(this.testee()).mimeMessageStore();
        MimeMessagePartsId partsId = (MimeMessagePartsId)mimeMessageStore.save((Object)message).block();
        StatementRecorder statementRecorder = new StatementRecorder();
        cassandra.getConf().recordStatements(statementRecorder);
        mimeMessageStore.read((Object)partsId).block();
        Assertions.assertThat((List)statementRecorder.listExecutedStatements(StatementRecorder.Selector.preparedStatementStartingWith((String)"SELECT data FROM blob_cache"))).hasSize(1);
    }

    @Nested
    class MetricsTest {
        MetricsTest() {
        }

        @Test
        void readBlobStoreCacheWithNoneDefaultBucketNameShouldNotImpact() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(TEST_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            CachedBlobStoreTest.this.testee.read(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            CachedBlobStoreTest.this.testee.read(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            SoftAssertions.assertSoftly(softly -> {
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).describedAs("blobStoreCacheMisses", new Object[0])).isEqualTo(0);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(0);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).describedAs("blobStoreCacheLatency", new Object[0])).hasSize(0);
            });
        }

        @Test
        void readBlobStoreWithNoneDefaultBucketNameShouldRecordByBackendLatency() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(TEST_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            CachedBlobStoreTest.this.testee.read(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            CachedBlobStoreTest.this.testee.read(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            SoftAssertions.assertSoftly(softly -> ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(2));
        }

        @Test
        void readBytesWithNoneDefaultBucketNameShouldNotImpact() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(TEST_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            SoftAssertions.assertSoftly(softly -> {
                ((AbstractIntegerAssert)Assertions.assertThat((int)CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).describedAs("blobStoreCacheMisses", new Object[0])).isEqualTo(0);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(0);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).describedAs("blobStoreCacheLatency", new Object[0])).hasSize(0);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(2);
            });
        }

        @Test
        void readBytesWithNoneDefaultBucketNameShouldPublishBackendTimerMetrics() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(TEST_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(TEST_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            SoftAssertions.assertSoftly(softly -> ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(2));
        }

        @Test
        void readBlobStoreCacheShouldPublishTimerMetrics() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            SoftAssertions.assertSoftly(softly -> ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).describedAs("blobStoreCacheLatency", new Object[0])).hasSize(2));
        }

        @Test
        void readBytesCacheShouldPublishTimerMetrics() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            SoftAssertions.assertSoftly(softly -> {
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).describedAs("blobStoreCacheLatency", new Object[0])).hasSize(2);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(2);
            });
        }

        @Test
        void readBytesShouldPublishBackendTimerMetricsForBigBlobs() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.backend.save(DEFAULT_BUCKETNAME, BlobStoreContract.ELEVEN_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            SoftAssertions.assertSoftly(softly -> ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(2));
        }

        @Test
        void readInputStreamShouldPublishBackendTimerForBigBlobs() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.backend.save(DEFAULT_BUCKETNAME, BlobStoreContract.ELEVEN_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            SoftAssertions.assertSoftly(softly -> ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(2));
        }

        @Test
        void readBytesShouldNotIncreaseCacheCounterForBigBlobs() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.backend.save(DEFAULT_BUCKETNAME, BlobStoreContract.ELEVEN_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            SoftAssertions.assertSoftly(softly -> {
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).describedAs("blobStoreCacheMisses", new Object[0])).isEqualTo(0);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(0);
            });
        }

        @Test
        void readInputStreamShouldNotIncreaseCacheCounterForBigBlobs() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.backend.save(DEFAULT_BUCKETNAME, BlobStoreContract.ELEVEN_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId);
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId);
            SoftAssertions.assertSoftly(softly -> {
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).describedAs("blobStoreCacheMisses", new Object[0])).isEqualTo(0);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(0);
            });
        }

        @Test
        void readBytesShouldRecordDistinctTimingsWhenRepeatAndBackendRead() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, BlobStoreContract.ELEVEN_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Duration delay = Duration.ofMillis(500L);
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).then(Mono.delay((Duration)delay)).repeat(2L).blockLast();
            ((AbstractCollectionAssert)Assertions.assertThat((Collection)CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).hasSize(3)).allSatisfy(timing -> Assertions.assertThat((Duration)timing).isLessThan((Comparable)delay));
        }

        @Test
        void readBytesShouldRecordDistinctTimingsWhenRepeat() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Duration delay = Duration.ofMillis(500L);
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).then(Mono.delay((Duration)delay)).repeat(2L).blockLast();
            ((AbstractCollectionAssert)Assertions.assertThat((Collection)CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).hasSize(3)).allSatisfy(timing -> Assertions.assertThat((Duration)timing).isLessThan((Comparable)delay));
        }

        @Test
        void readBlobStoreCacheShouldCountWhenHit() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            Assertions.assertThat((int)CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).isEqualTo(2);
        }

        @Test
        void readBytesCacheShouldCountWhenHit() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Assertions.assertThat((int)CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).isEqualTo(2);
        }

        @Test
        void readBlobStoreCacheShouldCountWhenMissed() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.backend.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.cache.remove(blobId)).block();
            CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE);
            Assertions.assertThat((int)CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).isEqualTo(1);
        }

        @Test
        void readBytesCacheShouldCountWhenMissed() {
            BlobId blobId = (BlobId)Mono.from((Publisher)CachedBlobStoreTest.this.testee.save(DEFAULT_BUCKETNAME, APPROXIMATELY_FIVE_KILOBYTES, BlobStore.StoragePolicy.SIZE_BASED)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.cache.remove(blobId)).block();
            Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, blobId, BlobStore.StoragePolicy.HIGH_PERFORMANCE)).block();
            Assertions.assertThat((int)CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).isEqualTo(1);
        }

        @Test
        void metricsShouldNotWorkExceptLatencyWhenReadNonExistingBlob() {
            SoftAssertions.assertSoftly(softly -> {
                softly.assertThatThrownBy(() -> CachedBlobStoreTest.this.testee.read(DEFAULT_BUCKETNAME, new TestBlobId.Factory().randomId(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).isInstanceOf(ObjectNotFoundException.class);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).describedAs("blobStoreCacheMisses", new Object[0])).isEqualTo(0);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(0);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).describedAs("blobStoreCacheLatency", new Object[0])).hasSize(1);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(1);
            });
        }

        @Test
        void metricsShouldNotWorkExceptLatencyWhenReadNonExistingBlobAsBytes() {
            SoftAssertions.assertSoftly(softly -> {
                softly.assertThatThrownBy(() -> Mono.from((Publisher)CachedBlobStoreTest.this.testee.readBytes(DEFAULT_BUCKETNAME, new TestBlobId.Factory().randomId(), BlobStore.StoragePolicy.HIGH_PERFORMANCE)).blockOptional()).isInstanceOf(ObjectNotFoundException.class);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheMisses")).describedAs("blobStoreCacheMisses", new Object[0])).isEqualTo(0);
                ((IntegerAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.countFor("blobStoreCacheHits")).describedAs("blobStoreCacheHits", new Object[0])).isEqualTo(0);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreCacheLatency")).describedAs("blobStoreCacheLatency", new Object[0])).hasSize(1);
                ((CollectionAssert)softly.assertThat(CachedBlobStoreTest.this.metricFactory.executionTimesFor("blobStoreBackEndLatency")).describedAs("blobStoreBackEndLatency", new Object[0])).hasSize(1);
            });
        }
    }
}

