/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.backup;

import com.google.common.collect.ImmutableList;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.UUID;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.zip.ExtraFieldUtils;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipExtraField;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.io.IOUtils;
import org.apache.james.junit.TemporaryFolderExtension;
import org.apache.james.mailbox.backup.Directory;
import org.apache.james.mailbox.backup.ZipAssert;
import org.apache.james.mailbox.backup.zip.SizeExtraField;
import org.apache.james.mailbox.backup.zip.UidExtraField;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(value={TemporaryFolderExtension.class})
class ZipAssertTest {
    private static final String ENTRY_NAME = "entryName";
    private static final String ENTRY_NAME_2 = "entryName2";
    private static final String DIRECTORY_NAME = "folder/";
    private static final String STRING_ENTRY_CONTENT = "abcdefghijkl";
    private static final String STRING_ENTRY_CONTENT_2 = "mnopqrstuvwxyz";
    private static final byte[] ENTRY_CONTENT = "abcdefghijkl".getBytes(StandardCharsets.UTF_8);
    private static final byte[] ENTRY_CONTENT_2 = "mnopqrstuvwxyz".getBytes(StandardCharsets.UTF_8);
    private static final SizeExtraField EXTRA_FIELD = new SizeExtraField(42L);
    private static final AbstractMap.SimpleImmutableEntry<String, byte[]> ENTRY = new AbstractMap.SimpleImmutableEntry<String, byte[]>("entryName", ENTRY_CONTENT);
    private static final AbstractMap.SimpleImmutableEntry<String, byte[]> ENTRY_2 = new AbstractMap.SimpleImmutableEntry<String, byte[]>("entryName2", ENTRY_CONTENT_2);
    private File destination;
    private File destination2;

    ZipAssertTest() {
    }

    private static ZipFile zipFile(File destination, ZipEntryWithContent ... entries) throws Exception {
        try (ZipArchiveOutputStream archiveOutputStream = new ZipArchiveOutputStream(destination);){
            for (ZipEntryWithContent entry : entries) {
                ZipArchiveEntry archiveEntry = (ZipArchiveEntry)archiveOutputStream.createArchiveEntry(new File(UUID.randomUUID().toString()), entry.name);
                entry.extraFields.forEach(arg_0 -> ((ZipArchiveEntry)archiveEntry).addExtraField(arg_0));
                archiveOutputStream.putArchiveEntry((ArchiveEntry)archiveEntry);
                IOUtils.copy((InputStream)entry.content, (OutputStream)archiveOutputStream);
                archiveOutputStream.closeArchiveEntry();
            }
            archiveOutputStream.finish();
        }
        return new ZipFile(destination);
    }

    @BeforeEach
    void beforeEach(TemporaryFolderExtension.TemporaryFolder temporaryFolder) throws Exception {
        this.destination = File.createTempFile("backup-test", ".zip", temporaryFolder.getTempDir());
        this.destination2 = File.createTempFile("backup-test2", ".zip", temporaryFolder.getTempDir());
        ExtraFieldUtils.register(SizeExtraField.class);
        ExtraFieldUtils.register(UidExtraField.class);
    }

    @SafeVarargs
    private ZipFile buildZipFile(AbstractMap.SimpleImmutableEntry<String, byte[]> ... entries) throws Exception {
        try (ZipArchiveOutputStream archiveOutputStream = new ZipArchiveOutputStream(this.destination);){
            for (AbstractMap.SimpleImmutableEntry<String, byte[]> entry : entries) {
                ZipArchiveEntry archiveEntry = (ZipArchiveEntry)archiveOutputStream.createArchiveEntry(new File("any"), entry.getKey());
                archiveOutputStream.putArchiveEntry((ArchiveEntry)archiveEntry);
                IOUtils.copy((InputStream)new ByteArrayInputStream(entry.getValue()), (OutputStream)archiveOutputStream);
                archiveOutputStream.closeArchiveEntry();
            }
            archiveOutputStream.finish();
        }
        return new ZipFile(this.destination);
    }

    @Test
    void hasNoEntryShouldNotThrowWhenEmpty() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(new AbstractMap.SimpleImmutableEntry[0]);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).hasNoEntry()).doesNotThrowAnyException();
        }
    }

    @Test
    void hasNoEntryShouldThrowWhenNotEmpty() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).hasNoEntry()).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsOnlyEntriesMatchingShouldNotThrowWhenBothEmpty() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(new AbstractMap.SimpleImmutableEntry[0]);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(new ZipAssert.EntryChecks[0])).doesNotThrowAnyException();
        }
    }

    @Test
    void containsOnlyEntriesMatchingShouldNotThrowWhenRightOrder() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME), ZipAssert.EntryChecks.hasName(ENTRY_NAME_2))).doesNotThrowAnyException();
        }
    }

    @Test
    void hasNameShouldThrowWhenWrongName() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME_2))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void isDirectoryShouldThrowWhenNotADirectory() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).isDirectory())).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void isDirectoryShouldNotThrowWhenDirectory() throws Exception {
        try (ZipArchiveOutputStream archiveOutputStream = new ZipArchiveOutputStream(this.destination);){
            ZipArchiveEntry archiveEntry = (ZipArchiveEntry)archiveOutputStream.createArchiveEntry((File)new Directory("any"), DIRECTORY_NAME);
            archiveOutputStream.putArchiveEntry((ArchiveEntry)archiveEntry);
            archiveOutputStream.closeArchiveEntry();
            archiveOutputStream.finish();
        }
        try (ZipFile zipFile = new ZipFile(this.destination);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(DIRECTORY_NAME).isDirectory())).doesNotThrowAnyException();
        }
    }

    @Test
    void containsOnlyEntriesMatchingShouldNotThrowWhenWrongOrder() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME_2), ZipAssert.EntryChecks.hasName(ENTRY_NAME))).doesNotThrowAnyException();
        }
    }

    @Test
    void containsOnlyEntriesMatchingShouldThrowWhenExpectingMoreEntries() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME), ZipAssert.EntryChecks.hasName(ENTRY_NAME_2), ZipAssert.EntryChecks.hasName("extraEntry"))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsOnlyEntriesMatchingShouldThrowWhenExpectingLessEntries() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsExactlyEntriesMatchingShouldNotThrowWhenBothEmpty() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(new AbstractMap.SimpleImmutableEntry[0]);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsExactlyEntriesMatching(new ZipAssert.EntryChecks[0])).doesNotThrowAnyException();
        }
    }

    @Test
    void containsExactlyEntriesMatchingShouldThrowWhenWrongOrder() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsExactlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME_2), ZipAssert.EntryChecks.hasName(ENTRY_NAME))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsExactlyEntriesMatchingShouldNotThrowWhenRightOrder() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsExactlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME), ZipAssert.EntryChecks.hasName(ENTRY_NAME_2))).doesNotThrowAnyException();
        }
    }

    @Test
    void containsExactlyEntriesMatchingShouldThrowWhenExpectingMoreEntries() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsExactlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME), ZipAssert.EntryChecks.hasName(ENTRY_NAME_2), ZipAssert.EntryChecks.hasName("extraEntry"))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsExactlyEntriesMatchingShouldThrowWhenExpectingLessEntries() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY, ENTRY_2);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsExactlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void hasStringContentShouldNotThrowWhenIdentical() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).hasStringContent(STRING_ENTRY_CONTENT))).doesNotThrowAnyException();
        }
    }

    @Test
    void hasStringContentShouldThrowWhenDifferent() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).hasStringContent(STRING_ENTRY_CONTENT_2))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsOnlyExtraFieldsShouldNotThrowWhenBothEmpty() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).containsExtraFields(new ZipExtraField[0]))).doesNotThrowAnyException();
        }
    }

    @Test
    void containsOnlyExtraFieldsShouldThrowWhenMissingExpectedField() throws Exception {
        try (ZipFile zipFile = this.buildZipFile(ENTRY);){
            Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).containsExtraFields(new ZipExtraField[]{EXTRA_FIELD}))).isInstanceOf(AssertionError.class);
        }
    }

    @Test
    void containsOnlyExtraFieldsShouldNotThrowWhenUnexpectedField() throws Exception {
        try (ZipArchiveOutputStream archiveOutputStream = new ZipArchiveOutputStream(this.destination);){
            ZipArchiveEntry archiveEntry = (ZipArchiveEntry)archiveOutputStream.createArchiveEntry(new File("any"), ENTRY_NAME);
            archiveEntry.addExtraField((ZipExtraField)EXTRA_FIELD);
            archiveOutputStream.putArchiveEntry((ArchiveEntry)archiveEntry);
            IOUtils.copy((InputStream)new ByteArrayInputStream(ENTRY_CONTENT), (OutputStream)archiveOutputStream);
            archiveOutputStream.closeArchiveEntry();
            archiveOutputStream.finish();
        }
        try (ZipFile zipFile = new ZipFile(this.destination);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).containsExtraFields(new ZipExtraField[0]))).doesNotThrowAnyException();
        }
    }

    @Test
    void containsOnlyExtraFieldsShouldNotThrowWhenContainingExpectedExtraFields() throws Exception {
        try (ZipArchiveOutputStream archiveOutputStream = new ZipArchiveOutputStream(this.destination);){
            ZipArchiveEntry archiveEntry = (ZipArchiveEntry)archiveOutputStream.createArchiveEntry(new File("any"), ENTRY_NAME);
            archiveEntry.addExtraField((ZipExtraField)EXTRA_FIELD);
            archiveOutputStream.putArchiveEntry((ArchiveEntry)archiveEntry);
            IOUtils.copy((InputStream)new ByteArrayInputStream(ENTRY_CONTENT), (OutputStream)archiveOutputStream);
            archiveOutputStream.closeArchiveEntry();
            archiveOutputStream.finish();
        }
        try (ZipFile zipFile = new ZipFile(this.destination);){
            Assertions.assertThatCode(() -> ZipAssert.assertThatZip(zipFile).containsOnlyEntriesMatching(ZipAssert.EntryChecks.hasName(ENTRY_NAME).containsExtraFields(new ZipExtraField[]{EXTRA_FIELD}))).doesNotThrowAnyException();
        }
    }

    static class ZipEntryWithContent {
        private final String name;
        private final InputStream content;
        private final ImmutableList<ZipExtraField> extraFields;

        static Builder.RequireName entryBuilder() {
            return name -> content -> new Builder.ReadyToBuild(name, content);
        }

        ZipEntryWithContent(String name, InputStream content, ImmutableList<ZipExtraField> extraFields) {
            this.name = name;
            this.content = content;
            this.extraFields = extraFields;
        }

        static class Builder {
            Builder() {
            }

            static class ReadyToBuild {
                private final String name;
                private final byte[] content;
                private final ImmutableList.Builder<ZipExtraField> extraFieldBuilder;

                ReadyToBuild(String name, byte[] content) {
                    this.name = name;
                    this.content = content;
                    this.extraFieldBuilder = new ImmutableList.Builder();
                }

                ReadyToBuild addField(ZipExtraField zipExtraField) {
                    this.extraFieldBuilder.add((Object)zipExtraField);
                    return this;
                }

                public ZipEntryWithContent build() {
                    return new ZipEntryWithContent(this.name, new ByteArrayInputStream(this.content), (ImmutableList<ZipExtraField>)this.extraFieldBuilder.build());
                }
            }

            @FunctionalInterface
            static interface RequireContent {
                public ReadyToBuild content(byte[] var1);
            }

            @FunctionalInterface
            static interface RequireName {
                public RequireContent name(String var1);
            }
        }
    }

    @Nested
    class AllSatisfies {
        AllSatisfies() {
        }

        @Test
        void allSatisfiesShouldNotThrowWhenNoEntries() throws Exception {
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, new ZipEntryWithContent[0]);){
                Assertions.assertThatCode(() -> ZipAssert.assertThatZip(assertedZipFile).allSatisfies(entry -> entry.hasStringContent("sub string"))).doesNotThrowAnyException();
            }
        }

        @Test
        void allSatisfiesShouldNotThrowWhenAllEntriesMatchAssertion() throws Exception {
            ZipEntryWithContent firstEntry = ZipEntryWithContent.entryBuilder().name("entry 1").content(ENTRY_CONTENT).build();
            ZipEntryWithContent secondEntry = ZipEntryWithContent.entryBuilder().name("entry 2").content(ENTRY_CONTENT).build();
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, firstEntry, secondEntry);){
                Assertions.assertThatCode(() -> ZipAssert.assertThatZip(assertedZipFile).allSatisfies(entry -> entry.hasStringContent(ZipAssertTest.STRING_ENTRY_CONTENT))).doesNotThrowAnyException();
            }
        }

        @Test
        void allSatisfiesShouldNotThrowWhenAllEntriesMatchAllAssertions() throws Exception {
            UidExtraField zipExtraField = new UidExtraField(1L);
            ZipEntryWithContent firstEntry = ZipEntryWithContent.entryBuilder().name("entry 1").content(ENTRY_CONTENT).addField((ZipExtraField)zipExtraField).build();
            ZipEntryWithContent secondEntry = ZipEntryWithContent.entryBuilder().name("entry 2").content(ENTRY_CONTENT).addField((ZipExtraField)zipExtraField).build();
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, firstEntry, secondEntry);){
                Assertions.assertThatCode(() -> ZipAssert.assertThatZip(assertedZipFile).allSatisfies(entry -> entry.hasStringContent(ZipAssertTest.STRING_ENTRY_CONTENT)).allSatisfies(entry -> entry.containsExtraFields(new ZipExtraField[]{zipExtraField}))).doesNotThrowAnyException();
            }
        }

        @Test
        void allSatisfiesShouldThrowWhenNotAllEntriesMatchAssertion() throws Exception {
            ZipEntryWithContent firstEntry = ZipEntryWithContent.entryBuilder().name("entry 1").content(ENTRY_CONTENT).build();
            ZipEntryWithContent secondEntry = ZipEntryWithContent.entryBuilder().name("entry 2").content(ENTRY_CONTENT).build();
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, firstEntry, secondEntry);){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).allSatisfies(entry -> ZipAssert.EntryChecks.hasName("entry 1"))).isInstanceOf(AssertionError.class);
            }
        }
    }

    @Nested
    class HasEntriesSize {
        HasEntriesSize() {
        }

        @Test
        void hasEntriesSizeShouldNotThrowWhenExpectingSizeEqualsEntriesSize() throws Exception {
            ZipEntryWithContent firstEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build();
            ZipEntryWithContent secondEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME_2).content(ENTRY_CONTENT).build();
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, firstEntry, secondEntry);){
                Assertions.assertThatCode(() -> ZipAssert.assertThatZip(assertedZipFile).hasEntriesSize(2)).doesNotThrowAnyException();
            }
        }

        @Test
        void hasEntriesSizeShouldNotThrowWhenNoEntriesAndExpectingSizeIsZero() throws Exception {
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, new ZipEntryWithContent[0]);){
                Assertions.assertThatCode(() -> ZipAssert.assertThatZip(assertedZipFile).hasEntriesSize(0)).doesNotThrowAnyException();
            }
        }

        @Test
        void hasEntriesSizeShouldThrowWhenExpectingSizeIsNegative() throws Exception {
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, new ZipEntryWithContent[0]);){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasEntriesSize(-1)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasEntriesSizeShouldThrowWhenExpectingSizeDoesntEqualsEntriesSize() throws Exception {
            ZipEntryWithContent firstEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build();
            ZipEntryWithContent secondEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME_2).content(ENTRY_CONTENT).build();
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, firstEntry, secondEntry);){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasEntriesSize(3)).isInstanceOf(AssertionError.class);
            }
        }
    }

    @Nested
    class HasSameContentTest {
        HasSameContentTest() {
        }

        @Test
        void hasSameContentShouldThrowWhenExpectedZipFileIsNull() throws Exception {
            try (ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build());){
                ZipFile expectedZipFile = null;
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenAssertedZipFileIsNull() throws Exception {
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build());){
                ZipFile assertedZipFile = null;
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenAssertedZipFileHasDifferentSizeWithExpectedZipFile() throws Exception {
            ZipEntryWithContent sameEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build();
            ZipEntryWithContent additionalEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME_2).content(ENTRY_CONTENT).build();
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, sameEntry);
                 ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination2, sameEntry, additionalEntry);){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenAssertedEntriesHaveDifferentContent() throws Exception {
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build());
                 ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination2, ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT_2).build());){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenAssertedEntriesHaveDifferentNames() throws Exception {
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).build());
                 ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination2, ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME_2).content(ENTRY_CONTENT).build());){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenEntryHasDifferentExtraFieldsSize() throws Exception {
            ZipEntryWithContent expectedEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).addField((ZipExtraField)new UidExtraField(1L)).addField((ZipExtraField)new UidExtraField(2L)).build();
            ZipEntryWithContent assertedEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).addField((ZipExtraField)new UidExtraField(1L)).build();
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, expectedEntry);
                 ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination2, assertedEntry);){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenEntryHasSameExtraFieldsSizeButDifferentOrder() throws Exception {
            ZipEntryWithContent expectedEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).addField((ZipExtraField)new UidExtraField(1L)).addField((ZipExtraField)new SizeExtraField(2L)).build();
            ZipEntryWithContent assertedEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).addField((ZipExtraField)new SizeExtraField(2L)).addField((ZipExtraField)new UidExtraField(1L)).build();
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, expectedEntry);
                 ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination2, assertedEntry);){
                Assertions.assertThatThrownBy(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).isInstanceOf(AssertionError.class);
            }
        }

        @Test
        void hasSameContentShouldThrowWhenEntryHasSameExtraFieldsSizeAndOrder() throws Exception {
            ZipEntryWithContent expectedEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).addField((ZipExtraField)new UidExtraField(1L)).addField((ZipExtraField)new SizeExtraField(2L)).build();
            ZipEntryWithContent assertedEntry = ZipEntryWithContent.entryBuilder().name(ZipAssertTest.ENTRY_NAME).content(ENTRY_CONTENT).addField((ZipExtraField)new UidExtraField(1L)).addField((ZipExtraField)new SizeExtraField(2L)).build();
            try (ZipFile expectedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination, expectedEntry);
                 ZipFile assertedZipFile = ZipAssertTest.zipFile(ZipAssertTest.this.destination2, assertedEntry);){
                Assertions.assertThatCode(() -> ZipAssert.assertThatZip(assertedZipFile).hasSameContentWith(expectedZipFile)).doesNotThrowAnyException();
            }
        }
    }
}

