/*
 * Decompiled with CFR 0.152.
 */
package tigase.cert;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.Security;
import java.security.SignatureException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertSelector;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Cipher;
import javax.security.auth.x500.X500Principal;
import tigase.annotations.TigaseDeprecated;
import tigase.cert.CertCheckResult;
import tigase.cert.CertificateEntry;
import tigase.cert.CertificateGenerator;
import tigase.cert.CertificateGeneratorFactory;
import tigase.cert.RSAPrivateKeyDecoder;
import tigase.util.Algorithms;
import tigase.util.Base64;

public abstract class CertificateUtil {
    protected static final byte[] ID_ON_XMPPADDR = new byte[]{6, 8, 43, 6, 1, 5, 5, 7, 8, 5};
    private static final String BEGIN_CERT = "-----BEGIN CERTIFICATE-----";
    private static final String BEGIN_KEY = "-----BEGIN PRIVATE KEY-----";
    private static final String BEGIN_RSA_KEY = "-----BEGIN RSA PRIVATE KEY-----";
    private static final String ENCRIPT_TEST = "--encript-test";
    private static final String ENCRIPT_TEST_SHORT = "-et";
    private static final String END_CERT = "-----END CERTIFICATE-----";
    private static final String END_KEY = "-----END PRIVATE KEY-----";
    private static final String END_RSA_KEY = "-----END RSA PRIVATE KEY-----";
    private static final String KEY_PAIR = "--key-pair";
    private static final String KEY_PAIR_SHORT = "-kp";
    private static final String LOAD_CERT = "--load-cert";
    private static final String LOAD_CERT_SHORT = "-lc";
    private static final String LOAD_DER_PRIVATE_KEY = "--load-der-priv-key";
    private static final String LOAD_DER_PRIVATE_KEY_SHORT = "-ldpk";
    private static final System.Logger log = System.getLogger(CertificateUtil.class.getName());
    private static final String PRINT_PROVIDERS = "--print-providers";
    private static final String PRINT_PROVIDERS_SHORT = "-pp";
    private static final String PRINT_SERVICES = "--print-services";
    private static final String PRINT_SERVICES_SHORT = "-ps";
    private static final String SELF_SIGNED_CERT = "--self-signed-cert";
    private static final String SELF_SIGNED_CERT_SHORT = "-ssc";
    private static final String STORE_CERT = "--store-cert";
    private static final String IPv4_IPv6_PATTERN = "^(((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?))|([0-9a-fA-F:]{2,}(:([0-9]{1,3}\\.){3}[0-9]{1,3})?))$";
    private static final String STORE_CERT_SHORT = "-sc";

    private static int calculateLength(byte[] buffer, int start) throws ArrayIndexOutOfBoundsException {
        log.log(System.Logger.Level.DEBUG, "calculating length, buffer: {0}, start: {1}", new String(buffer), start);
        int offset = start + 1;
        int b = buffer[offset] & 0xFF;
        if (b < 128) {
            return b;
        }
        int result = 0;
        ++offset;
        int len = b - 128;
        for (int i = 0; i < len; ++i) {
            b = buffer[i + offset] & 0xFF;
            result = (result << 8) + b;
        }
        return result;
    }

    private static final int calculateOffset(byte[] buffer, int offset) throws ArrayIndexOutOfBoundsException {
        log.log(System.Logger.Level.DEBUG, "calculating offset, buffer: {0}, start: {1}", new String(buffer), offset);
        int b = buffer[offset + 1] & 0xFF;
        if (b < 128) {
            return offset + 2;
        }
        int len = b - 128;
        return offset + len + 2;
    }

    public static KeyPair createKeyPair(int size, String password) throws NoSuchAlgorithmException {
        log.log(System.Logger.Level.TRACE, "creating KeyPair, size: {0}, password: {1}", size, password);
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA");
        keyPairGenerator.initialize(size);
        KeyPair keyPair = keyPairGenerator.genKeyPair();
        log.log(System.Logger.Level.TRACE, "creating KeyPair, KeyPairGenerator: {0}, keyPair: {1}", keyPairGenerator, keyPair);
        return keyPair;
    }

    @Deprecated
    @TigaseDeprecated(since="8.0.0", removeIn="7.3.0", note="Use method with keyPairSupplier")
    public static X509Certificate createSelfSignedCertificate(String email, String domain, String organizationUnit, String organization, String city, String state, String country, KeyPair keyPair) throws CertificateException, IOException, NoSuchAlgorithmException, InvalidKeyException, NoSuchProviderException, SignatureException {
        return CertificateGeneratorFactory.getGenerator().generateSelfSignedCertificate(email, domain, organizationUnit, organization, city, state, country, keyPair);
    }

    public static CertificateEntry createSelfSignedCertificate(String email, String domain, String organizationUnit, String organization, String city, String state, String country, KeyPairSupplier keyPairSupplier) throws CertificateException, IOException, NoSuchAlgorithmException, InvalidKeyException, NoSuchProviderException, SignatureException {
        CertificateGenerator generator = CertificateGeneratorFactory.getGenerator();
        KeyPair keyPair = keyPairSupplier.get();
        try {
            return generator.generateSelfSignedCertificateEntry(email, domain, organizationUnit, organization, city, state, country, keyPair);
        }
        catch (GeneralSecurityException e) {
            throw new CertificateException(e);
        }
    }

    private static void encriptTest() throws Exception {
        System.out.print("Generating key pair...");
        System.out.flush();
        KeyPair keyPair = CertificateUtil.createKeyPair(1024, "secret");
        System.out.println(" done.");
        byte[] inputText = "Encription test...".getBytes();
        Cipher cipher = Cipher.getInstance("RSA");
        System.out.println("Encripting text: " + new String(inputText));
        cipher.init(1, keyPair.getPublic());
        byte[] cipherText = cipher.doFinal(inputText);
        System.out.println("Encripted text: " + Algorithms.bytesToHex(cipherText));
        cipher.init(2, keyPair.getPrivate());
        byte[] plainText = cipher.doFinal(cipherText);
        System.out.println("Decripted text: " + new String(plainText));
    }

    public static String exportToPemFormat(CertificateEntry entry) throws CertificateEncodingException {
        String b64;
        byte[] bytes;
        log.log(System.Logger.Level.TRACE, "exportToPemFormat cert, entry: {0}", entry);
        StringBuilder sb = new StringBuilder(4096);
        if (entry.getCertChain() != null && entry.getCertChain().length > 0) {
            bytes = entry.getCertChain()[0].getEncoded();
            b64 = Base64.encode(bytes);
            sb.append(BEGIN_CERT).append('\n').append(b64).append('\n').append(END_CERT).append('\n');
        }
        if (entry.getPrivateKey() != null) {
            bytes = entry.getPrivateKey().getEncoded();
            b64 = Base64.encode(bytes);
            sb.append(BEGIN_KEY).append('\n').append(b64).append('\n').append(END_KEY).append('\n');
        }
        if (entry.getCertChain() != null && entry.getCertChain().length > 1) {
            for (int i = 1; i < entry.getCertChain().length; ++i) {
                byte[] bytes2 = entry.getCertChain()[i].getEncoded();
                String b642 = Base64.encode(bytes2);
                sb.append(BEGIN_CERT).append('\n').append(b642).append('\n').append(END_CERT).append('\n');
            }
        }
        log.log(System.Logger.Level.TRACE, "exportToPemFormat cert, string: {0}", sb.toString());
        return sb.toString();
    }

    protected static String extractCN(X500Principal principal) {
        String[] dd;
        for (String string : dd = principal.getName("RFC2253").split(",")) {
            if (!string.toLowerCase().startsWith("cn=")) continue;
            log.log(System.Logger.Level.TRACE, "extractCN, principal: {0}, result: {1}", principal, string.substring(3));
            return string.substring(3);
        }
        return null;
    }

    private static String extractValue(byte[] buffer, byte[] id) {
        log.log(System.Logger.Level.DEBUG, "extracting value, buffer: {0}, id: {1}", new String(buffer), new String(id));
        try {
            if (buffer[0] != 48) {
                return null;
            }
            int len = CertificateUtil.calculateLength(buffer, 0);
            int offset = CertificateUtil.calculateOffset(buffer, 0);
            for (int i = 0; i < id.length; ++i) {
                int j = offset + i;
                if (j >= len) {
                    return null;
                }
                if (id[i] == buffer[j]) continue;
                return null;
            }
            int valStart = offset + id.length;
            int pos = CertificateUtil.calculateOffset(buffer, valStart);
            while (pos < buffer.length) {
                byte d = buffer[pos];
                int cmp = CertificateUtil.calculateOffset(buffer, pos);
                int l = CertificateUtil.calculateLength(buffer, pos);
                if (d == 12 || d == 22) {
                    return new String(buffer, cmp, l);
                }
                pos = cmp;
            }
            return null;
        }
        catch (ArrayIndexOutOfBoundsException e) {
            return null;
        }
    }

    public static List<String> extractXmppAddrs(X509Certificate x509Certificate) {
        log.log(System.Logger.Level.TRACE, "extractXmppAddrs, x509Certificate: {0}", x509Certificate.toString());
        ArrayList<String> result = new ArrayList<String>();
        try {
            Collection<List<?>> altNames = x509Certificate.getSubjectAlternativeNames();
            if (altNames == null) {
                return result;
            }
            for (List<?> item : altNames) {
                byte[] buffer;
                String jid;
                Integer type = (Integer)item.get(0);
                if (type != 0 || (jid = CertificateUtil.extractValue(buffer = (byte[])item.get(1), ID_ON_XMPPADDR)) == null) continue;
                result.add(jid);
            }
            log.log(System.Logger.Level.TRACE, "extractXmppAddrs, result: {0}", result);
            return result;
        }
        catch (Exception e) {
            return result;
        }
    }

    public static List<String> getCertAltCName(X509Certificate cert) {
        log.log(System.Logger.Level.TRACE, "getCertAltCName, x509Certificate: {0}", cert.toString());
        try {
            Collection<List<?>> subjectAlternativeNames = cert.getSubjectAlternativeNames();
            ArrayList<String> result = new ArrayList<String>();
            if (subjectAlternativeNames != null && subjectAlternativeNames.size() > 0) {
                for (List<?> list : subjectAlternativeNames) {
                    if (!list.get(0).equals(2)) continue;
                    result.add(list.get(1).toString());
                }
            }
            log.log(System.Logger.Level.DEBUG, "Certificate alternative names: {0}", result);
            return result;
        }
        catch (CertificateParsingException e) {
            return Collections.emptyList();
        }
    }

    public static String getCertCName(X509Certificate cert) {
        String[] all;
        log.log(System.Logger.Level.TRACE, "getCertCName, X509Certificate: {0}", cert);
        X500Principal princ = cert.getSubjectX500Principal();
        String name = princ.getName();
        for (String n : all = name.split(",")) {
            String[] ns = n.trim().split("=");
            if (!ns[0].equals("CN")) continue;
            log.log(System.Logger.Level.DEBUG, "Certificate DN: {0}", ns[1]);
            return ns[1];
        }
        return null;
    }

    public static String getCertificateBasicInfo(Certificate cert) {
        return CertificateUtil.getCertificateBasicInfo(new StringBuilder(), cert).toString();
    }

    public static StringBuilder getCertificateBasicInfo(StringBuilder sb, Certificate cert) {
        if (cert instanceof X509Certificate) {
            X509Certificate certX509 = (X509Certificate)cert;
            sb.append("CN: ").append(CertificateUtil.getCertCName(certX509)).append('\n');
            List<String> certAltCName = CertificateUtil.getCertAltCName(certX509);
            if (certAltCName != null && !certAltCName.isEmpty()) {
                sb.append('\t').append("alt: ").append(certAltCName).append('\n');
            }
            sb.append('\t').append("Issuer: ").append(certX509.getIssuerDN()).append('\n');
            sb.append('\t').append("Not Before: ").append(certX509.getNotBefore()).append('\n');
            sb.append('\t').append("Not After: ").append(certX509.getNotAfter()).append('\n');
            try {
                sb.append('\t').append("Fingerprint: ").append(CertificateUtil.getCertificateFingerprint(certX509)).append('\n');
            }
            catch (Exception e) {
                log.log(System.Logger.Level.WARNING, "Could not calculate fingerprint", (Throwable)e);
            }
            CertificateUtil.getCertificateSerialNumber(certX509).ifPresent(serialNumber -> sb.append('\t').append("SerialNumber: [").append(serialNumber.toString(16)).append("]\n"));
            sb.append('\n');
        }
        return sb;
    }

    public static String getCertificateFingerprint(Certificate cert) throws CertificateEncodingException, NoSuchAlgorithmException {
        MessageDigest md = MessageDigest.getInstance("SHA-1");
        md.update(cert.getEncoded());
        return Algorithms.bytesToHex(md.digest());
    }

    public static Optional<BigInteger> getCertificateSerialNumber(Certificate cert) {
        BigInteger serialNumber = null;
        if (cert instanceof X509Certificate) {
            X509Certificate certX509 = (X509Certificate)cert;
            serialNumber = certX509.getSerialNumber();
        }
        return Optional.ofNullable(serialNumber);
    }

    private static Certificate getRootCertificateCertificate(List<Certificate> certs) {
        Certificate rt = null;
        for (Certificate x509Certificate : certs) {
            Principal s;
            Principal i = ((X509Certificate)x509Certificate).getIssuerDN();
            if (!i.equals(s = ((X509Certificate)x509Certificate).getSubjectDN())) continue;
            rt = x509Certificate;
        }
        return rt;
    }

    public static boolean isExpired(X509Certificate cert) {
        try {
            cert.checkValidity();
            return false;
        }
        catch (Exception e) {
            return true;
        }
    }

    public static boolean isSelfSigned(X509Certificate cert) {
        boolean result = cert.getIssuerDN().equals(cert.getSubjectDN());
        if (result) {
            log.log(System.Logger.Level.INFO, "Self-signed certificate for domain: {0}", cert.getSubjectDN());
        }
        log.log(System.Logger.Level.TRACE, "isSelfSigned, result: {0}, X509Certificate: {1}", result, cert);
        return result;
    }

    private static void keyPairTest() throws Exception {
        System.out.print("Generating key pair...");
        System.out.flush();
        KeyPair keyPair = CertificateUtil.createKeyPair(1024, "secret");
        System.out.println(" done, private key: " + String.valueOf(keyPair.getPrivate()) + ", public key: " + String.valueOf(keyPair.getPublic()));
    }

    public static CertificateEntry loadCertificate(File file) throws FileNotFoundException, IOException, CertificateException, NoSuchAlgorithmException, InvalidKeySpecException {
        return CertificateUtil.parseCertificate(new FileReader(file));
    }

    @Deprecated
    @TigaseDeprecated(since="4.2.0", note="Method loads only single certificate; use other #loadCertificate() methods")
    public static CertificateEntry loadCertificate(byte[] bytes) throws CertificateException, NoSuchProviderException {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        Certificate certificate = cf.generateCertificate(bais);
        CertificateEntry entry = new CertificateEntry();
        entry.setCertChain(new Certificate[]{certificate});
        return entry;
    }

    public static CertificateEntry loadCertificate(String file) throws FileNotFoundException, IOException, CertificateException, NoSuchAlgorithmException, InvalidKeySpecException {
        return CertificateUtil.loadCertificate(new File(file));
    }

    public static PrivateKey loadPrivateKeyFromDER(File file) throws FileNotFoundException, IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        log.log(System.Logger.Level.INFO, "loadPrivateKeyFromDER, file: {0}", file);
        DataInputStream dis = new DataInputStream(new FileInputStream(file));
        byte[] privKeyBytes = new byte[(int)file.length()];
        dis.read(privKeyBytes);
        dis.close();
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        PKCS8EncodedKeySpec privSpec = new PKCS8EncodedKeySpec(privKeyBytes);
        RSAPrivateKey privKey = (RSAPrivateKey)keyFactory.generatePrivate(privSpec);
        return privKey;
    }

    public static void main(String[] args) throws Exception {
        Level lvl = Level.FINE;
        Logger julLogger = Logger.getLogger("tigase.cert");
        julLogger.setLevel(lvl);
        ConsoleHandler consoleHandler = new ConsoleHandler();
        consoleHandler.setLevel(lvl);
        julLogger.addHandler(consoleHandler);
        if (args != null && args.length > 0) {
            String file;
            if (args[0].equals(PRINT_PROVIDERS) || args[0].equals(PRINT_PROVIDERS_SHORT)) {
                CertificateUtil.printProviders(false);
            }
            if (args[0].equals(PRINT_SERVICES) || args[0].equals(PRINT_SERVICES_SHORT)) {
                CertificateUtil.printProviders(true);
            }
            if (args[0].equals(KEY_PAIR) || args[0].equals(KEY_PAIR_SHORT)) {
                CertificateUtil.keyPairTest();
            }
            if (args[0].equals(ENCRIPT_TEST) || args[0].equals(ENCRIPT_TEST_SHORT)) {
                CertificateUtil.encriptTest();
            }
            if (args[0].equals(SELF_SIGNED_CERT) || args[0].equals(SELF_SIGNED_CERT_SHORT)) {
                CertificateUtil.selfSignedCertTest();
            }
            if (args[0].equals(LOAD_CERT) || args[0].equals(LOAD_CERT_SHORT)) {
                file = args[1];
                CertificateEntry ce = CertificateUtil.loadCertificate(file);
                boolean basic = args.length == 3 && "-simple".equals(args[2]);
                System.out.println(ce.toString(basic));
                ArrayList<Certificate> certs = new ArrayList<Certificate>(Arrays.asList(ce.getCertChain()));
                if (CertificateUtil.getRootCertificateCertificate(certs) == null) {
                    System.out.println("Can't find root certificate in chain!");
                    for (Certificate x509Certificate : certs) {
                        Principal i = ((X509Certificate)x509Certificate).getIssuerDN();
                        Principal s = ((X509Certificate)x509Certificate).getSubjectDN();
                        System.out.println(String.valueOf(s) + " ~ ISSUED BY: " + String.valueOf(i));
                    }
                }
                CertificateUtil.sort(ce.getCertChain());
            }
            if (args[0].equals(STORE_CERT) || args[0].equals(STORE_CERT_SHORT)) {
                file = args[1];
                String cname = args[2];
                String email = "artur.hefczyc@tigase.org";
                String domain = cname != null ? cname : "tigase.org";
                String ou = "XMPP Service";
                String o = "Tigase.org";
                String l = "Cambourne";
                String st = "Cambridgeshire";
                String c = "UK";
                CertificateEntry entry = CertificateUtil.createSelfSignedCertificate(email, domain, ou, o, l, st, c, () -> CertificateUtil.createKeyPair(1024, "secret"));
                CertificateUtil.storeCertificate(file, entry);
            }
            if (args[0].equals(LOAD_DER_PRIVATE_KEY) || args[0].equals(LOAD_DER_PRIVATE_KEY_SHORT)) {
                file = args[1];
                PrivateKey key = CertificateUtil.loadPrivateKeyFromDER(new File(file));
                System.out.println(key.toString());
            }
        } else {
            CertificateUtil.printHelp();
        }
    }

    public static boolean match(String hostname, String altName) {
        int dotIdx;
        if (hostname == null || hostname.isEmpty() || altName == null || altName.isEmpty()) {
            return false;
        }
        String normalizedAltName = altName.toLowerCase(Locale.US);
        if (!normalizedAltName.contains("*")) {
            return hostname.equals(normalizedAltName);
        }
        if (normalizedAltName.startsWith("*.") && hostname.regionMatches(0, normalizedAltName, 2, normalizedAltName.length() - 2)) {
            return true;
        }
        int asteriskIdx = normalizedAltName.indexOf(42);
        if (asteriskIdx > (dotIdx = normalizedAltName.indexOf(46))) {
            return false;
        }
        if (!hostname.regionMatches(0, normalizedAltName, 0, asteriskIdx)) {
            return false;
        }
        int suffixLength = normalizedAltName.length() - (asteriskIdx + 1);
        int suffixStart = hostname.length() - suffixLength;
        if (hostname.indexOf(46, asteriskIdx) < suffixStart) {
            return false;
        }
        return hostname.regionMatches(suffixStart, normalizedAltName, asteriskIdx + 1, suffixLength);
    }

    public static CertificateEntry parseCertificate(Reader data) throws IOException, CertificateException, NoSuchAlgorithmException, InvalidKeySpecException {
        String line;
        BufferedReader br = new BufferedReader(data);
        StringBuilder sb = new StringBuilder(4096);
        ArrayList<X509Certificate> certs = new ArrayList<X509Certificate>();
        PrivateKey privateKey = null;
        boolean addToBuffer = false;
        while ((line = br.readLine()) != null) {
            byte[] bytes;
            if (line.contains(BEGIN_CERT) || line.contains(BEGIN_KEY) || line.contains(BEGIN_RSA_KEY)) {
                addToBuffer = true;
                continue;
            }
            if (line.contains(END_CERT)) {
                addToBuffer = false;
                bytes = Base64.decode(sb.toString());
                ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                while (bais.available() > 0) {
                    Certificate cert = cf.generateCertificate(bais);
                    certs.add((X509Certificate)cert);
                }
                sb = new StringBuilder(4096);
                continue;
            }
            if (line.contains(END_KEY)) {
                addToBuffer = false;
                bytes = Base64.decode(sb.toString());
                PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(bytes);
                privateKey = CertificateUtil.generateKeyWithFallback(keySpec);
                log.log(System.Logger.Level.TRACE, "parseCertificate, privateKey: {0}", privateKey);
                sb = new StringBuilder(4096);
                continue;
            }
            if (line.contains(END_RSA_KEY)) {
                addToBuffer = false;
                bytes = Base64.decode(sb.toString());
                RSAPrivateKeyDecoder decoder = new RSAPrivateKeyDecoder(bytes);
                privateKey = decoder.getPrivateKey();
                log.log(System.Logger.Level.TRACE, "parseCertificate, privateKey: {0}", privateKey);
                sb = new StringBuilder(4096);
                continue;
            }
            if (!addToBuffer) continue;
            sb.append(line);
        }
        CertificateEntry entry = new CertificateEntry();
        entry.setCertChain(certs.toArray(new Certificate[certs.size()]));
        entry.setPrivateKey(privateKey);
        log.log(System.Logger.Level.TRACE, "parseCertificate, entry: {0}", entry);
        return entry;
    }

    private static PrivateKey generateKeyWithFallback(KeySpec keySpec) throws NoSuchAlgorithmException, InvalidKeySpecException {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePrivate(keySpec);
        }
        catch (InvalidKeySpecException e) {
            KeyFactory keyFactory = KeyFactory.getInstance("EC");
            return keyFactory.generatePrivate(keySpec);
        }
    }

    private static void printHelp() {
        System.out.println(CertificateUtil.class.getName() + " test code.");
        System.out.println("You can run following tests:");
        System.out.println(" --print-providers | -pp - prints all supported providers");
        System.out.println(" --print-services | -ps - print all supported services");
        System.out.println(" --key-pair | -kp - generate a key pair and print the result");
        System.out.println(" --encript-test | -et - encript simple text with public key, decript with private");
        System.out.println(" --self-signed-cert | -ssc - generate self signed certificate");
        System.out.println(" --load-cert file.pem | -lc file.pem - load certificate from file");
        System.out.println(" --store-cert file.pem | -sc file.pem - generate self-signed certificate and save it to the given pem file");
        System.out.println(" --load-der-priv-key | -ldpk file.der - load private key from DER file.");
    }

    private static void printProviders(boolean includeServices) {
        Provider[] providers = Security.getProviders();
        if (providers != null && providers.length > 0) {
            for (Provider provider : providers) {
                System.out.println(provider.getName() + "\t" + provider.getInfo());
                if (!includeServices) continue;
                for (Provider.Service service : provider.getServices()) {
                    System.out.println("\t" + service.getAlgorithm());
                }
            }
        } else {
            System.out.println("No security providers found!");
        }
    }

    public static Certificate[] removeRootCACertificate(Certificate[] certChain) {
        return (Certificate[])Arrays.stream(certChain).filter(X509Certificate.class::isInstance).map(X509Certificate.class::cast).filter(cert -> {
            boolean[] keyUsage = cert.getKeyUsage();
            return keyUsage == null || !keyUsage[5] || !CertificateUtil.isSelfSigned(cert);
        }).toArray(Certificate[]::new);
    }

    private static void selfSignedCertTest() throws Exception {
        KeyPair keyPair = CertificateUtil.createKeyPair(1024, "secret");
        String email = "artur.hefczyc@tigase.org";
        String domain = "tigase.org";
        String ou = "XMPP Service";
        String o = "Tigase.org";
        String l = "Cambourne";
        String st = "Cambridgeshire";
        String c = "UK";
        System.out.println("Creating self-signed certificate for issuer: " + domain);
        CertificateEntry entry = CertificateUtil.createSelfSignedCertificate(email, domain, ou, o, l, st, c, () -> keyPair);
        X509Certificate cert = (X509Certificate)entry.getCertChain()[0];
        System.out.print("Checking certificate validity today...");
        System.out.flush();
        cert.checkValidity();
        System.out.println(" done.");
        System.out.print("Checking certificate validity yesterday...");
        System.out.flush();
        try {
            cert.checkValidity(new Date(System.currentTimeMillis() - 86400000L));
            System.out.println(" error.");
        }
        catch (CertificateNotYetValidException e) {
            System.out.println(" not valid!");
        }
        System.out.print("Verifying certificate with public key...");
        System.out.flush();
        if (entry.getKeyPair().isPresent()) {
            cert.verify(entry.getKeyPair().get().getPublic());
        } else {
            System.out.println(" KeyPair is missing");
        }
        System.out.println(" done.");
        System.out.println(cert.toString());
    }

    public static Certificate[] sort(Certificate[] chain) {
        List<Certificate> res = CertificateUtil.sort(new ArrayList<Certificate>(Arrays.asList(chain)));
        return res.toArray(new Certificate[res.size()]);
    }

    public static List<Certificate> sort(List<Certificate> certs) {
        Certificate rt = CertificateUtil.getRootCertificateCertificate(certs);
        if (rt == null) {
            throw new RuntimeException("Can't find root certificate in chain!");
        }
        ArrayList<Certificate> res = new ArrayList<Certificate>();
        certs.remove(rt);
        res.add(rt);
        while (!certs.isEmpty()) {
            boolean found = false;
            for (Certificate x509Certificate : certs) {
                Principal i = ((X509Certificate)x509Certificate).getIssuerDN();
                if (!i.equals(((X509Certificate)rt).getSubjectDN())) continue;
                rt = x509Certificate;
                found = true;
                break;
            }
            if (found) {
                certs.remove(rt);
                res.add(0, rt);
                continue;
            }
            throw new RuntimeException("Can't find certificate " + String.valueOf(((X509Certificate)rt).getSubjectDN()) + " in chain. Verify that all entries are correct and match against each other!");
        }
        return res;
    }

    public static void storeCertificate(String file, CertificateEntry entry) throws CertificateEncodingException, IOException {
        log.log(System.Logger.Level.TRACE, "storeCertificate, file: {0}, entry: {1}", file, entry);
        String pemFormat = CertificateUtil.exportToPemFormat(entry);
        File f = new File(file);
        if (f.exists()) {
            f.renameTo(new File(file + ".bak"));
        }
        FileWriter fw = new FileWriter(f, false);
        fw.write(pemFormat);
        fw.close();
    }

    public static CertCheckResult validateCertificate(Certificate[] chain, KeyStore trustKeystore, boolean revocationEnabled) throws NoSuchAlgorithmException, KeyStoreException, InvalidAlgorithmParameterException, CertificateException {
        if (log.isLoggable(System.Logger.Level.TRACE)) {
            log.log(System.Logger.Level.TRACE, "Validating cert: {0}, chain size: {1}, trustKeystore size: {2}, revocationEnabled: {3}", ((X509Certificate)chain[0]).getSubjectDN(), chain.length, trustKeystore.size(), revocationEnabled);
        }
        CertPathValidator certPathValidator = CertPathValidator.getInstance(CertPathValidator.getDefaultType());
        X509CertSelector selector = new X509CertSelector();
        PKIXBuilderParameters params = new PKIXBuilderParameters(trustKeystore, (CertSelector)selector);
        params.setRevocationEnabled(false);
        List<Certificate> certList = Arrays.asList(chain);
        CertPath certPath = CertificateFactory.getInstance("X.509").generateCertPath(certList);
        try {
            certPathValidator.validate(certPath, params);
            return CertCheckResult.trusted;
        }
        catch (CertPathValidatorException ex) {
            if (CertificateUtil.isExpired((X509Certificate)chain[0])) {
                return CertCheckResult.expired;
            }
            if (chain.length == 1 && CertificateUtil.isSelfSigned((X509Certificate)chain[0])) {
                return CertCheckResult.self_signed;
            }
            return CertCheckResult.untrusted;
        }
    }

    public static boolean verifyCertificateForDomain(X509Certificate cert, String hostname) throws CertificateParsingException {
        if (hostname.matches(IPv4_IPv6_PATTERN)) {
            return CertificateUtil.verifyCertificateForIp(hostname, cert);
        }
        return CertificateUtil.verifyCertificateForHostname(hostname, cert);
    }

    protected static boolean verifyCertificateForHostname(String hostname, X509Certificate x509Certificate) throws CertificateParsingException {
        X500Principal principal;
        String cn;
        log.log(System.Logger.Level.TRACE, "verifyCertificateForHostname, hostname: {0}, x509Certificate: {1}", hostname, x509Certificate);
        boolean altNamePresents = false;
        Collection<List<?>> altNames = x509Certificate.getSubjectAlternativeNames();
        if (altNames != null) {
            for (List<?> entry : altNames) {
                Integer altNameType = (Integer)entry.get(0);
                if (altNameType != 2) continue;
                altNamePresents = true;
                String altName = (String)entry.get(1);
                if (!CertificateUtil.match(hostname, altName)) continue;
                return true;
            }
        }
        if (!altNamePresents && (cn = CertificateUtil.extractCN(principal = x509Certificate.getSubjectX500Principal())) != null) {
            return CertificateUtil.match(hostname, cn);
        }
        return false;
    }

    protected static boolean verifyCertificateForIp(String ipAddr, X509Certificate x509Certificate) throws CertificateParsingException {
        log.log(System.Logger.Level.TRACE, "verifyCertificateForIp, ipAddr: {0}, x509Certificate: {1}", ipAddr, x509Certificate);
        for (List<?> entry : x509Certificate.getSubjectAlternativeNames()) {
            String altName;
            Integer altNameType = (Integer)entry.get(0);
            if (!(altNameType != 7 ? altNameType == 2 && ipAddr.equals(altName = (String)entry.get(1)) : ipAddr.equalsIgnoreCase(altName = (String)entry.get(1)))) continue;
            return true;
        }
        return false;
    }

    public static interface KeyPairSupplier {
        public KeyPair get() throws NoSuchAlgorithmException;
    }
}

