/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.apns;

import org.junit.Test;
import tigase.push.Device;
import tigase.push.PlainNotification;
import tigase.push.api.INotification;
import tigase.push.api.IPushSettings;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.logging.LogManager;

import static org.junit.Assert.*;

public class APNSProviderTest {

	// old tokens are still active even if I do not have the device for it?
	private final String account = System.getProperty("account", "test@example.com");
	private final String sender = System.getProperty("sender");
	private final String deviceId = System.getProperty("deviceId");
	private final String pushkitDeviceId = System.getProperty("pushKitDeviceId");
	private final String encryptionKeyId = System.getProperty("encryptionKeyId");
	private final String encryptionKey = System.getProperty("encryptionKey");
	private final String teamId = System.getProperty("teamId");

	@Test
	public void testProvider() throws NoSuchFieldException, IllegalAccessException, InterruptedException, IOException,
									  TigaseStringprepException, ExecutionException {
		if (encryptionKey == null) {
			return;
		}
		
		LogManager.getLogManager()
				.readConfiguration(new ByteArrayInputStream(
						"handlers=java.util.logging.ConsoleHandler\ntigase.push.apns.level=ALL\ntigase.jaxmpp.j2se.connectors.socket.SocketConnector.level=ALL\njava.util.logging.ConsoleHandler.level=FINEST"
								.getBytes()));

		APNsBinaryApiProvider provider = new APNsBinaryApiProvider();

		Field f = provider.getClass().getDeclaredField("encryptionKeyId");
		f.setAccessible(true);
		f.set(provider, encryptionKeyId);
		f = provider.getClass().getDeclaredField("encryptionKey");
		f.setAccessible(true);
		f.set(provider, encryptionKey);
		f = provider.getClass().getDeclaredField("teamId");
		f.setAccessible(true);
		f.set(provider, teamId);
		f = provider.getClass().getDeclaredField("fallbackToSandbox");
		f.setAccessible(true);
		f.set(provider, true);
		f = provider.getClass().getDeclaredField("apnsTopic");
		f.setAccessible(true);
		f.set(provider, "org.tigase.tygrys.custom.atom");
		f = provider.getClass().getDeclaredField("notificationType");
		f.setAccessible(true);
		f.set(provider, APNsBinaryApiProvider.NotificationType.automatic);

		provider.beanConfigurationChanged(Arrays.asList("encryptionKey"));


//		Notification notification = new Notification(BareJID.bareJIDInstance("test@example.com"), 5l,
//													 JID.jidInstance("sender@example.com/resource-1"),
//													 "Some very important message " + UUID.randomUUID().toString(), null);

		IPushSettings.IDevice device = new Device("dummy", deviceId, null);
		for (int i=0; i<1; i++) {
			PlainNotification plainNotification = new PlainNotification(BareJID.bareJIDInstance(account), INotification.Priority.high, null,
																		null,
																		"New message 123!", null);
//			PlainNotification plainNotification = new PlainNotification(BareJID.bareJIDInstance("andrzej@hi-low.eu"), INotification.Priority.high, 1l,
//																		JID.jidInstance("home@hi-low.eu"),
//																		"Some very important message " + UUID.randomUUID().toString(), null);

			CompletableFuture<String> future = provider.pushNotification(device, plainNotification);
			String apnsId = future.get();
			assertNotNull("APNS ID is NULL!", apnsId);
			System.out.println("send push notification with id " + apnsId);
		}
		Thread.sleep(1000);
	}

	@Test
	public void testProvider_jingle() throws NoSuchFieldException, IllegalAccessException, InterruptedException, IOException,
									  TigaseStringprepException, ExecutionException {
		if (encryptionKey == null) {
			return;
		}

		LogManager.getLogManager()
				.readConfiguration(new ByteArrayInputStream(
						"handlers=java.util.logging.ConsoleHandler\ntigase.push.apns.level=ALL\ntigase.jaxmpp.j2se.connectors.socket.SocketConnector.level=ALL\njava.util.logging.ConsoleHandler.level=FINEST"
								.getBytes()));

		APNsBinaryApiProvider provider = new APNsBinaryApiProvider();

		Field f = provider.getClass().getDeclaredField("encryptionKeyId");
		f.setAccessible(true);
		f.set(provider, encryptionKeyId);
		f = provider.getClass().getDeclaredField("encryptionKey");
		f.setAccessible(true);
		f.set(provider, encryptionKey);
		f = provider.getClass().getDeclaredField("teamId");
		f.setAccessible(true);
		f.set(provider, teamId);
		f = provider.getClass().getDeclaredField("fallbackToSandbox");
		f.setAccessible(true);
		f.set(provider, true);
		f = provider.getClass().getDeclaredField("apnsTopic");
		f.setAccessible(true);
		f.set(provider, "org.tigase.tygrys.custom.atom");
		f = provider.getClass().getDeclaredField("notificationType");
		f.setAccessible(true);
		f.set(provider, APNsBinaryApiProvider.NotificationType.automatic);

		provider.beanConfigurationChanged(Arrays.asList("encryptionKey"));

		
		IPushSettings.IDevice device = new Device("dummy", deviceId, pushkitDeviceId);
		for (int i=0; i<1; i++) {
			PlainNotification plainNotification = new PlainNotification(BareJID.bareJIDInstance(account), INotification.Priority.high, null,
																		JID.jidInstance(sender),
																		null, null, null,
																		new INotification.Jingle("test", List.of("audio", "video")), null);

			CompletableFuture<String> future = provider.pushNotification(device, plainNotification);
			String apnsId = future.get();
			assertNotNull("APNS ID is NULL!", apnsId);
			System.out.println("send push notification with id " + apnsId);
		}
		Thread.sleep(1000);
	}

	@Test
	public void payloadGenerationTest() throws TigaseStringprepException, NoSuchFieldException, IllegalAccessException {
		APNsBinaryApiProvider provider = new APNsBinaryApiProvider();

		PlainNotification plainNotification = new PlainNotification(BareJID.bareJIDInstance(account), INotification.Priority.high, null,
																	null,
																	"New message 123!", null);
		ApnsPayload.Builder builder = ApnsPayload.newBuilder();
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.automatic);
		ApnsPayload payload = builder.build();
		assertEquals(plainNotification.getAccount().toString(), payload.customField("account"));
		assertEquals(plainNotification.getMessageCount(), payload.customField("unread-messages"));
		assertEquals(plainNotification.getLastMessageBody(), payload.customField("body"));
		assertNull(payload.badge());

		builder = ApnsPayload.newBuilder();
		Field f = APNsBinaryApiProvider.class.getDeclaredField("includeAccountInPayload");
		f.setAccessible(true);
		f.set(provider, false);
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.automatic);
		payload = builder.build();
		assertNull(payload.customField("account"));
		assertEquals(plainNotification.getMessageCount(), payload.customField("unread-messages"));
		assertEquals(plainNotification.getLastMessageBody(), payload.customField("body"));
		assertNull(payload.badge());

		builder = ApnsPayload.newBuilder();
		f = APNsBinaryApiProvider.class.getDeclaredField("staticNotification");
		f.setAccessible(true);
		f.set(provider, true);
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationTitle");
		f.setAccessible(true);
		f.set(provider, "New message!");
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationBody");
		f.setAccessible(true);
		f.set(provider, "There is a new message available!");
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationBadge");
		f.setAccessible(true);
		f.set(provider, 1);
		assertFalse(builder.hasBody());
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.automatic);
		assertTrue(builder.hasBody());
		payload = builder.build();
		assertNull(payload.customField("account"));
		assertNull(payload.customField("unread-messages"));
		assertNull(payload.customField("body"));
		assertEquals("New message!", payload.title());
		assertEquals("There is a new message available!", payload.body());
		assertEquals((Integer) 1, payload.badge());
	}

	@Test
	public void payloadGenerationTest_SendNotificationToFetchOnly() throws TigaseStringprepException, NoSuchFieldException, IllegalAccessException {
		APNsBinaryApiProvider provider = new APNsBinaryApiProvider();

		PlainNotification plainNotification = new PlainNotification(BareJID.bareJIDInstance(account), INotification.Priority.high, null,
																	null,
																	"New message 123!", null);
		ApnsPayload.Builder builder = ApnsPayload.newBuilder();
		Field f = APNsBinaryApiProvider.class.getDeclaredField("includeAccountInPayload");
		f.setAccessible(true);
		f.set(provider, false);
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationType");
		f.setAccessible(true);
		f.set(provider, APNsBinaryApiProvider.NotificationType.background);
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.background);
		assertFalse(builder.hasBody());
		ApnsPayload payload = builder.build();
		assertNull(payload.customField("account"));
		assertEquals(plainNotification.getMessageCount(), payload.customField("unread-messages"));
		assertEquals(plainNotification.getLastMessageBody(), payload.customField("body"));
		assertNull(payload.badge());

		builder = ApnsPayload.newBuilder();
		f = APNsBinaryApiProvider.class.getDeclaredField("staticNotification");
		f.setAccessible(true);
		f.set(provider, true);
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationTitle");
		f.setAccessible(true);
		f.set(provider, "New message!");
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationBody");
		f.setAccessible(true);
		f.set(provider, "There is a new message available!");
		f = APNsBinaryApiProvider.class.getDeclaredField("notificationBadge");
		f.setAccessible(true);
		f.set(provider, 1);
		assertFalse(builder.hasBody());
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.background);
		assertFalse(builder.hasBody());
		payload = builder.build();
		assertNull(payload.customField("account"));
		assertNull(payload.customField("unread-messages"));
		assertNull(payload.customField("body"));
		assertNull(payload.title());
		assertNull(payload.body());
		assertNull(payload.badge());
	}

	@Test
	public void payloadGenerationTestAccountRemoved() throws TigaseStringprepException, NoSuchFieldException, IllegalAccessException {
		APNsBinaryApiProvider provider = new APNsBinaryApiProvider();

		PlainNotification plainNotification = new PlainNotification(BareJID.bareJIDInstance(account), INotification.Priority.high, null,
																	null,
																	null,null, "account-removed", null, null);
		ApnsPayload.Builder builder = ApnsPayload.newBuilder();
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.automatic);
		ApnsPayload payload = builder.build();
		assertEquals(plainNotification.getAccount().toString(), payload.customField("account"));
		assertEquals(plainNotification.getMessageCount(), payload.customField("unread-messages"));
		assertEquals(plainNotification.getLastMessageBody(), payload.customField("body"));
		assertEquals(plainNotification.getEventName(), payload.customField("event-name"));
		assertNull(payload.badge());

		builder = ApnsPayload.newBuilder();
		Field f = APNsBinaryApiProvider.class.getDeclaredField("includeAccountInPayload");
		f.setAccessible(true);
		f.set(provider, false);
		builder = provider.preparePayload(builder, plainNotification, APNsBinaryApiProvider.NotificationType.automatic);
		payload = builder.build();
		assertNull(payload.customField("account"));
		assertEquals(plainNotification.getMessageCount(), payload.customField("unread-messages"));
		assertEquals(plainNotification.getLastMessageBody(), payload.customField("body"));
		assertNull(payload.badge());
	}

}
