/*
 * halcyon-core
 * Copyright (C) 2018 Tigase, Inc. (office@tigase.com)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */
package tigase.halcyon.core.xmpp.modules.uniqueId

import tigase.halcyon.core.Context
import tigase.halcyon.core.builder.HalcyonConfigDsl
import tigase.halcyon.core.modules.*
import tigase.halcyon.core.xml.Element
import tigase.halcyon.core.xmpp.BareJID
import tigase.halcyon.core.xmpp.ErrorCondition
import tigase.halcyon.core.xmpp.XMPPException
import tigase.halcyon.core.xmpp.stanzas.Message

@HalcyonConfigDsl
interface UniqueStableStanzaIdModuleConfig

/**
 * Module is implementing Unique and Stable Stanza IDs ([XEP-0359](https://xmpp.org/extensions/xep-0359.html)).
 */
class UniqueStableStanzaIdModule(override val context: Context) : XmppModule,
    UniqueStableStanzaIdModuleConfig {

    /**
     * Module is implementing Unique and Stable Stanza IDs ([XEP-0359](https://xmpp.org/extensions/xep-0359.html)).
     */
    companion object : XmppModuleProvider<UniqueStableStanzaIdModule, UniqueStableStanzaIdModuleConfig> {

        const val XMLNS = "urn:xmpp:sid:0"
        override val TYPE = XMLNS

        override fun instance(context: Context): UniqueStableStanzaIdModule = UniqueStableStanzaIdModule(context)

        override fun configure(module: UniqueStableStanzaIdModule, cfg: UniqueStableStanzaIdModuleConfig.() -> Unit) =
            module.cfg()

        override fun doAfterRegistration(module: UniqueStableStanzaIdModule, moduleManager: ModulesManager) =
            moduleManager.registerOutgoingFilter(createFilter(module::beforeSend))


    }

    override val type = TYPE
    override val criteria: Criteria? = null
    override val features = arrayOf(XMLNS)

    override fun process(element: Element) = throw XMPPException(ErrorCondition.BadRequest)

    private fun beforeSend(element: Element?, chain: StanzaFilterChain) {
        if (element?.name == Message.NAME && element.attributes["id"] != null && element.getChildrenNS(
                "origin-id",
                XMLNS
            ) == null
        ) {
            element.add(tigase.halcyon.core.xml.element("origin-id") {
                xmlns = XMLNS
                attributes["id"] = element.attributes["id"]
            })
        }
        chain.doFilter(element)
    }

    fun getStanzaID(element: Element): String? {
        val jid = context.boundJID?.bareJID ?: return null
        return element.getStanzaIDBy(jid)
    }

}

/**
 * Returns Stanza ID generated by given XMPP entity.
 * @param by JabberID of creator of stanza ID.
 */
fun Element.getStanzaIDBy(by: BareJID): String? {
    val stanzaId = this.children.firstOrNull {
        it.name == "stanza-id" && it.xmlns == UniqueStableStanzaIdModule.XMLNS && it.attributes["by"] == by.toString()
    } ?: return null
    return stanzaId.attributes["id"]
}

/**
 * Returns Origin stanza ID.
 */
fun Element.getOriginID(): String? =
    this.getChildrenNS("origin-id", UniqueStableStanzaIdModule.XMLNS)?.attributes?.get("id")

