/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.lucene.search;

import com.github.fge.lambdas.Throwing;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import jakarta.annotation.PreDestroy;
import jakarta.inject.Inject;
import jakarta.mail.Flags;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;
import java.util.stream.Stream;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.SessionProvider;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.UnsupportedSearchException;
import org.apache.james.mailbox.lucene.search.LenientImapSearchAnalyzer;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.mailbox.model.MessageAttachmentMetadata;
import org.apache.james.mailbox.model.MessageId;
import org.apache.james.mailbox.model.MessageRange;
import org.apache.james.mailbox.model.SearchQuery;
import org.apache.james.mailbox.model.UpdatedFlags;
import org.apache.james.mailbox.store.MailboxSessionMapperFactory;
import org.apache.james.mailbox.store.mail.model.MailboxMessage;
import org.apache.james.mailbox.store.search.ListeningMessageSearchIndex;
import org.apache.james.mailbox.store.search.MessageSearchIndex;
import org.apache.james.mailbox.store.search.SearchUtil;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.dom.Header;
import org.apache.james.mime4j.dom.address.Address;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.dom.address.Group;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.dom.address.MailboxList;
import org.apache.james.mime4j.dom.field.DateTimeField;
import org.apache.james.mime4j.field.address.AddressFormatter;
import org.apache.james.mime4j.field.address.LenientAddressParser;
import org.apache.james.mime4j.message.SimpleContentHandler;
import org.apache.james.mime4j.parser.ContentHandler;
import org.apache.james.mime4j.parser.MimeStreamParser;
import org.apache.james.mime4j.stream.BodyDescriptor;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeConfig;
import org.apache.james.mime4j.util.MimeUtil;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.NumericDocValuesField;
import org.apache.lucene.document.SortedSetDocValuesField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.SortedSetSortField;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TermRangeQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopFieldDocs;
import org.apache.lucene.search.WildcardQuery;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.BytesRef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class LuceneMessageSearchIndex
extends ListeningMessageSearchIndex {
    private static final Logger log = LoggerFactory.getLogger(LuceneMessageSearchIndex.class);
    private static final Date MAX_DATE;
    private static final Date MIN_DATE;
    public static final org.apache.james.events.Group GROUP;
    private static final int DEFAULT_MAX_QUERY_RESULTS = 100000;
    @VisibleForTesting
    static final String ID_FIELD = "id";
    static final String UID_FIELD = "uid";
    private static final String HAS_ATTACHMENT_FIELD = "hasAttachment";
    static final String FLAGS_FIELD = "flags";
    private static final String SIZE_FIELD = "size";
    private static final String BODY_FIELD = "body";
    private static final String PREFIX_HEADER_FIELD = "header_";
    private static final String HEADERS_FIELD = "headers";
    private static final String MODSEQ_FIELD = "modSeq";
    private static final String THREAD_ID_FIELD = "threadId";
    private static final String TO_FIELD = "to";
    private static final String FIRST_TO_MAILBOX_NAME_FIELD = "firstToMailboxName";
    private static final String FIRST_TO_MAILBOX_DISPLAY_FIELD = "firstToMailboxDisplay";
    private static final String CC_FIELD = "cc";
    private static final String FIRST_CC_MAILBOX_NAME_FIELD = "firstCcMailboxName";
    private static final String FROM_FIELD = "from";
    private static final String FIRST_FROM_MAILBOX_NAME_FIELD = "firstFromMailboxName";
    private static final String FIRST_FROM_MAILBOX_DISPLAY_FIELD = "firstFromMailboxDisplay";
    private static final String BCC_FIELD = "bcc";
    private static final String BASE_SUBJECT_FIELD = "baseSubject";
    private static final String INTERNAL_DATE_FIELD_YEAR_RESOLUTION = "internaldateYearResolution";
    private static final String INTERNAL_DATE_FIELD_MONTH_RESOLUTION = "internaldateMonthResolution";
    private static final String INTERNAL_DATE_FIELD_DAY_RESOLUTION = "internaldateDayResolution";
    private static final String INTERNAL_DATE_FIELD_HOUR_RESOLUTION = "internaldateHourResolution";
    private static final String INTERNAL_DATE_FIELD_MINUTE_RESOLUTION = "internaldateMinuteResolution";
    private static final String INTERNAL_DATE_FIELD_SECOND_RESOLUTION = "internaldateSecondResolution";
    private static final String INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION = "internaldateMillisecondResolution";
    private static final String SAVE_DATE_FIELD_YEAR_RESOLUTION = "saveDateYearResolution";
    private static final String SAVE_DATE_FIELD_MONTH_RESOLUTION = "saveDateMonthResolution";
    private static final String SAVE_DATE_FIELD_DAY_RESOLUTION = "saveDateDayResolution";
    private static final String SAVE_DATE_FIELD_HOUR_RESOLUTION = "saveDateHourResolution";
    private static final String SAVE_DATE_FIELD_MINUTE_RESOLUTION = "saveDateMinuteResolution";
    private static final String SAVE_DATE_FIELD_SECOND_RESOLUTION = "saveDateSecondResolution";
    static final String MAILBOX_ID_FIELD = "mailboxid";
    private static final String USERS = "userSession";
    private static final String MESSAGE_ID_FIELD = "messageid";
    private static final String SENT_DATE_FIELD_YEAR_RESOLUTION = "sentdateYearResolution";
    private static final String SENT_DATE_FIELD_MONTH_RESOLUTION = "sentdateMonthResolution";
    private static final String SENT_DATE_FIELD_DAY_RESOLUTION = "sentdateDayResolution";
    private static final String SENT_DATE_FIELD_HOUR_RESOLUTION = "sentdateHourResolution";
    private static final String SENT_DATE_FIELD_MINUTE_RESOLUTION = "sentdateMinuteResolution";
    private static final String SENT_DATE_FIELD_SECOND_RESOLUTION = "sentdateSecondResolution";
    private static final String SENT_DATE_FIELD_MILLISECOND_RESOLUTION = "sentdateMillisecondResolution";
    private static final String SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION = "sentdateSort";
    private static final String MEDIA_TYPE_TEXT = "text";
    private static final String MEDIA_TYPE_MESSAGE = "message";
    private static final String DEFAULT_ENCODING = "US-ASCII";
    private static final SortField UID_SORT;
    private static final SortField UID_SORT_REVERSE;
    private static final SortField SIZE_SORT;
    private static final SortField SIZE_SORT_REVERSE;
    private static final SortField FIRST_CC_MAILBOX_SORT;
    private static final SortField FIRST_CC_MAILBOX_SORT_REVERSE;
    private static final SortField FIRST_TO_MAILBOX_SORT;
    private static final SortField FIRST_TO_MAILBOX_SORT_REVERSE;
    private static final SortField FIRST_FROM_MAILBOX_SORT;
    private static final SortField FIRST_FROM_MAILBOX_SORT_REVERSE;
    private static final SortField ARRIVAL_MAILBOX_SORT;
    private static final SortField ARRIVAL_MAILBOX_SORT_REVERSE;
    private static final SortField BASE_SUBJECT_SORT;
    private static final SortField BASE_SUBJECT_SORT_REVERSE;
    private static final SortField SENT_DATE_SORT;
    private static final SortField SENT_DATE_SORT_REVERSE;
    private final MailboxId.Factory mailboxIdFactory;
    private final MessageId.Factory messageIdFactory;
    @VisibleForTesting
    final IndexWriter writer;
    private final Directory directory;
    private int maxQueryResults = 100000;
    private boolean suffixMatch = false;

    @Inject
    public LuceneMessageSearchIndex(MailboxSessionMapperFactory factory, MailboxId.Factory mailboxIdFactory, Directory directory, MessageId.Factory messageIdFactory, SessionProvider sessionProvider) throws IOException {
        this(factory, mailboxIdFactory, directory, false, messageIdFactory, sessionProvider);
    }

    public LuceneMessageSearchIndex(MailboxSessionMapperFactory factory, MailboxId.Factory mailboxIdFactory, Directory directory, boolean dropIndexOnStart, MessageId.Factory messageIdFactory, SessionProvider sessionProvider) throws IOException {
        super(factory, (Set)ImmutableSet.of(), sessionProvider);
        this.mailboxIdFactory = mailboxIdFactory;
        this.messageIdFactory = messageIdFactory;
        this.directory = directory;
        try {
            this.writer = new IndexWriter(this.directory, this.createConfig(LenientImapSearchAnalyzer.INSTANCE, dropIndexOnStart));
        }
        catch (IndexFormatTooOldException e) {
            throw new RuntimeException("Old lucene index version detected, automatic migration is not supported. See https://github.com/james/james-project/blob/master/upgrade-instructions.md#james-4046-refactor-and-update-apache-james-mailbox-lucene for details", e);
        }
    }

    @PreDestroy
    public void close() throws IOException {
        log.trace("Closing Lucene index");
        this.writer.commit();
        this.writer.close();
    }

    public org.apache.james.events.Group getDefaultGroup() {
        return GROUP;
    }

    public EnumSet<MailboxManager.SearchCapabilities> getSupportedCapabilities(EnumSet<MailboxManager.MessageCapabilities> messageCapabilities) {
        return EnumSet.of(MailboxManager.SearchCapabilities.MultimailboxSearch);
    }

    public void setMaxQueryResults(int maxQueryResults) {
        this.maxQueryResults = maxQueryResults;
    }

    protected IndexWriterConfig createConfig(Analyzer analyzer, boolean dropIndexOnStart) {
        IndexWriterConfig config = new IndexWriterConfig(analyzer);
        if (dropIndexOnStart) {
            config.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
        } else {
            config.setOpenMode(IndexWriterConfig.OpenMode.CREATE_OR_APPEND);
        }
        return config;
    }

    public void setEnableSuffixMatch(boolean suffixMatch) {
        this.suffixMatch = suffixMatch;
    }

    public Flux<MessageUid> doSearch(MailboxSession session, org.apache.james.mailbox.model.Mailbox mailbox, SearchQuery searchQuery) throws MailboxException {
        Preconditions.checkArgument((session != null ? 1 : 0) != 0, (Object)"'session' is mandatory");
        return Flux.fromIterable(this.searchMultimap((Collection<MailboxId>)ImmutableList.of((Object)mailbox.getMailboxId()), searchQuery)).map(MessageSearchIndex.SearchResult::getMessageUid);
    }

    public Flux<MessageId> search(MailboxSession session, Collection<MailboxId> mailboxIds, SearchQuery searchQuery, long limit) throws MailboxException {
        Preconditions.checkArgument((session != null ? 1 : 0) != 0, (Object)"'session' is mandatory");
        if (mailboxIds.isEmpty()) {
            return Flux.empty();
        }
        return Flux.fromIterable((Iterable)((Iterable)this.searchMultimap(mailboxIds, searchQuery).stream().map(searchResult -> (MessageId)searchResult.getMessageId().get()).filter(SearchUtil.distinct()).limit(Long.valueOf(limit).intValue()).collect(ImmutableList.toImmutableList())));
    }

    private List<MessageSearchIndex.SearchResult> searchMultimap(Collection<MailboxId> mailboxIds, SearchQuery searchQuery) throws MailboxException {
        ImmutableList.Builder results = ImmutableList.builder();
        Query inMailboxes = this.buildQueryFromMailboxes(mailboxIds);
        try (DirectoryReader reader = DirectoryReader.open((IndexWriter)this.writer);){
            ScoreDoc[] sDocs;
            IndexSearcher searcher = new IndexSearcher((IndexReader)reader);
            BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
            queryBuilder.add(inMailboxes, BooleanClause.Occur.MUST);
            queryBuilder.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST_NOT);
            List crits = searchQuery.getCriteria();
            for (SearchQuery.Criterion crit : crits) {
                queryBuilder.add(this.createQuery(crit, inMailboxes, searchQuery.getRecentMessageUids()), BooleanClause.Occur.MUST);
            }
            TopFieldDocs docs = searcher.search((Query)queryBuilder.build(), this.maxQueryResults, this.createSort(searchQuery.getSorts()));
            for (ScoreDoc sDoc : sDocs = docs.scoreDocs) {
                Document doc = searcher.doc(sDoc.doc);
                MessageUid uid = MessageUid.of((long)doc.getField(UID_FIELD).numericValue().longValue());
                MailboxId mailboxId = this.mailboxIdFactory.fromString(doc.get(MAILBOX_ID_FIELD));
                Optional<MessageId> messageId = this.toMessageId(Optional.ofNullable(doc.get(MESSAGE_ID_FIELD)));
                results.add((Object)new MessageSearchIndex.SearchResult(messageId, mailboxId, uid));
            }
        }
        catch (IOException e) {
            throw new MailboxException("Unable to search the mailbox", (Throwable)e);
        }
        return results.build();
    }

    private Optional<MessageId> toMessageId(Optional<String> messageIdField) {
        if (messageIdField.isPresent()) {
            return Optional.of(this.messageIdFactory.fromString(messageIdField.get()));
        }
        return Optional.empty();
    }

    private Query buildQueryFromMailboxes(Collection<MailboxId> mailboxIds) {
        BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
        for (MailboxId id : mailboxIds) {
            String idAsString = id.serialize();
            queryBuilder.add((Query)new TermQuery(new Term(MAILBOX_ID_FIELD, idAsString)), BooleanClause.Occur.SHOULD);
        }
        return queryBuilder.build();
    }

    private Document createMessageDocument(MailboxSession session, final MailboxMessage membership) throws IOException, MimeException {
        String serializedThreadId;
        final Document doc = new Document();
        doc.add((IndexableField)new StringField(USERS, session.getUser().asString().toUpperCase(Locale.US), Field.Store.YES));
        doc.add((IndexableField)new StringField(MAILBOX_ID_FIELD, membership.getMailboxId().serialize().toUpperCase(Locale.US), Field.Store.YES));
        doc.add((IndexableField)new NumericDocValuesField(UID_FIELD, membership.getUid().asLong()));
        doc.add((IndexableField)new LongPoint(UID_FIELD, new long[]{membership.getUid().asLong()}));
        doc.add((IndexableField)new StoredField(UID_FIELD, membership.getUid().asLong()));
        doc.add((IndexableField)new StringField(HAS_ATTACHMENT_FIELD, Boolean.toString(LuceneMessageSearchIndex.hasAttachment(membership)), Field.Store.YES));
        String serializedMessageId = SearchUtil.getSerializedMessageIdIfSupportedByUnderlyingStorageOrNull((MailboxMessage)membership);
        if (serializedMessageId != null) {
            doc.add((IndexableField)new StringField(MESSAGE_ID_FIELD, serializedMessageId, Field.Store.YES));
        }
        if ((serializedThreadId = SearchUtil.getSerializedThreadIdIfSupportedByUnderlyingStorageOrNull((MailboxMessage)membership)) != null) {
            doc.add((IndexableField)new StringField(THREAD_ID_FIELD, serializedThreadId, Field.Store.YES));
        }
        doc.add((IndexableField)new StringField(ID_FIELD, membership.getMailboxId().serialize().toUpperCase(Locale.US) + "-" + Long.toString(membership.getUid().asLong()), Field.Store.YES));
        doc.add((IndexableField)new StringField(INTERNAL_DATE_FIELD_YEAR_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.YEAR), Field.Store.NO));
        doc.add((IndexableField)new StringField(INTERNAL_DATE_FIELD_MONTH_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.MONTH), Field.Store.NO));
        doc.add((IndexableField)new StringField(INTERNAL_DATE_FIELD_DAY_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.DAY), Field.Store.NO));
        doc.add((IndexableField)new StringField(INTERNAL_DATE_FIELD_HOUR_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.HOUR), Field.Store.NO));
        doc.add((IndexableField)new StringField(INTERNAL_DATE_FIELD_MINUTE_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.MINUTE), Field.Store.NO));
        doc.add((IndexableField)new StringField(INTERNAL_DATE_FIELD_SECOND_RESOLUTION, DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.SECOND), Field.Store.NO));
        doc.add((IndexableField)new NumericDocValuesField(INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION, Long.parseLong(DateTools.dateToString((Date)membership.getInternalDate(), (DateTools.Resolution)DateTools.Resolution.MILLISECOND))));
        membership.getSaveDate().ifPresent(saveDate -> {
            doc.add((IndexableField)new StringField(SAVE_DATE_FIELD_YEAR_RESOLUTION, DateTools.dateToString((Date)saveDate, (DateTools.Resolution)DateTools.Resolution.YEAR), Field.Store.NO));
            doc.add((IndexableField)new StringField(SAVE_DATE_FIELD_MONTH_RESOLUTION, DateTools.dateToString((Date)saveDate, (DateTools.Resolution)DateTools.Resolution.MONTH), Field.Store.NO));
            doc.add((IndexableField)new StringField(SAVE_DATE_FIELD_DAY_RESOLUTION, DateTools.dateToString((Date)saveDate, (DateTools.Resolution)DateTools.Resolution.DAY), Field.Store.NO));
            doc.add((IndexableField)new StringField(SAVE_DATE_FIELD_HOUR_RESOLUTION, DateTools.dateToString((Date)saveDate, (DateTools.Resolution)DateTools.Resolution.HOUR), Field.Store.NO));
            doc.add((IndexableField)new StringField(SAVE_DATE_FIELD_MINUTE_RESOLUTION, DateTools.dateToString((Date)saveDate, (DateTools.Resolution)DateTools.Resolution.MINUTE), Field.Store.NO));
            doc.add((IndexableField)new StringField(SAVE_DATE_FIELD_SECOND_RESOLUTION, DateTools.dateToString((Date)saveDate, (DateTools.Resolution)DateTools.Resolution.SECOND), Field.Store.NO));
        });
        doc.add((IndexableField)new LongPoint(SIZE_FIELD, new long[]{membership.getFullContentOctets()}));
        doc.add((IndexableField)new NumericDocValuesField(SIZE_FIELD, membership.getFullContentOctets()));
        SimpleContentHandler handler = new SimpleContentHandler(this){

            public void headers(Header header) {
                Date sentDate = null;
                String firstFromMailbox = "";
                String firstToMailbox = "";
                String firstCcMailbox = "";
                String firstFromDisplay = "";
                String firstToDisplay = "";
                for (Field f : header) {
                    String headerName = f.getName().toUpperCase(Locale.US);
                    String headerValue = f.getBody().toUpperCase(Locale.US);
                    String fullValue = f.toString().toUpperCase(Locale.US);
                    doc.add((IndexableField)new TextField(LuceneMessageSearchIndex.HEADERS_FIELD, fullValue, Field.Store.NO));
                    doc.add((IndexableField)new TextField(LuceneMessageSearchIndex.PREFIX_HEADER_FIELD + headerName, headerValue, Field.Store.NO));
                    if (f instanceof DateTimeField) {
                        DateTimeField dateTimeField = (DateTimeField)f;
                        sentDate = dateTimeField.getDate();
                    }
                    String field = null;
                    if ("To".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.TO_FIELD;
                    } else if ("From".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.FROM_FIELD;
                    } else if ("Cc".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.CC_FIELD;
                    } else if ("Bcc".equalsIgnoreCase(headerName)) {
                        field = LuceneMessageSearchIndex.BCC_FIELD;
                    }
                    if (field != null) {
                        AddressList aList = LenientAddressParser.DEFAULT.parseAddressList((CharSequence)MimeUtil.unfold((String)f.getBody()));
                        for (int i = 0; i < aList.size(); ++i) {
                            Address address = aList.get(i);
                            if (address instanceof Mailbox) {
                                Mailbox mailbox = (Mailbox)address;
                                String value = AddressFormatter.DEFAULT.encode(mailbox).toUpperCase(Locale.US);
                                doc.add((IndexableField)new TextField(field, value, Field.Store.NO));
                                if (i != 0) continue;
                                String mailboxAddress = SearchUtil.getMailboxAddress((Mailbox)mailbox);
                                String mailboxDisplay = SearchUtil.getDisplayAddress((Mailbox)mailbox);
                                if ("To".equalsIgnoreCase(headerName)) {
                                    firstToMailbox = mailboxAddress;
                                    firstToDisplay = mailboxDisplay;
                                    continue;
                                }
                                if ("From".equalsIgnoreCase(headerName)) {
                                    firstFromMailbox = mailboxAddress;
                                    firstFromDisplay = mailboxDisplay;
                                    continue;
                                }
                                if (!"Cc".equalsIgnoreCase(headerName)) continue;
                                firstCcMailbox = mailboxAddress;
                                continue;
                            }
                            if (!(address instanceof Group)) continue;
                            MailboxList mList = ((Group)address).getMailboxes();
                            for (int a = 0; a < mList.size(); ++a) {
                                Mailbox mailbox = mList.get(a);
                                String value = AddressFormatter.DEFAULT.encode(mailbox).toUpperCase(Locale.US);
                                doc.add((IndexableField)new TextField(field, value, Field.Store.NO));
                                if (i != 0 || a != 0) continue;
                                String mailboxAddress = SearchUtil.getMailboxAddress((Mailbox)mailbox);
                                String mailboxDisplay = SearchUtil.getDisplayAddress((Mailbox)mailbox);
                                if ("To".equalsIgnoreCase(headerName)) {
                                    firstToMailbox = mailboxAddress;
                                    firstToDisplay = mailboxDisplay;
                                    continue;
                                }
                                if ("From".equalsIgnoreCase(headerName)) {
                                    firstFromMailbox = mailboxAddress;
                                    firstFromDisplay = mailboxDisplay;
                                    continue;
                                }
                                if (!"Cc".equalsIgnoreCase(headerName)) continue;
                                firstCcMailbox = mailboxAddress;
                            }
                        }
                        doc.add((IndexableField)new TextField(field, headerValue, Field.Store.NO));
                        continue;
                    }
                    if (!headerName.equalsIgnoreCase("Subject")) continue;
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.BASE_SUBJECT_FIELD, SearchUtil.getBaseSubject((String)headerValue), Field.Store.YES));
                    doc.add((IndexableField)new SortedSetDocValuesField(LuceneMessageSearchIndex.BASE_SUBJECT_FIELD, new BytesRef((CharSequence)SearchUtil.getBaseSubject((String)headerValue))));
                }
                if (sentDate == null) {
                    sentDate = membership.getInternalDate();
                } else {
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_YEAR_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.YEAR), Field.Store.NO));
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_MONTH_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MONTH), Field.Store.NO));
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_DAY_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.DAY), Field.Store.NO));
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_HOUR_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.HOUR), Field.Store.NO));
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_MINUTE_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MINUTE), Field.Store.NO));
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_SECOND_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.SECOND), Field.Store.NO));
                    doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.SENT_DATE_FIELD_MILLISECOND_RESOLUTION, DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MILLISECOND), Field.Store.NO));
                }
                doc.removeField(LuceneMessageSearchIndex.SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION);
                doc.add((IndexableField)new NumericDocValuesField(LuceneMessageSearchIndex.SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION, Long.parseLong(DateTools.dateToString((Date)sentDate, (DateTools.Resolution)DateTools.Resolution.MILLISECOND))));
                doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.FIRST_FROM_MAILBOX_NAME_FIELD, firstFromMailbox, Field.Store.YES));
                doc.add((IndexableField)new SortedSetDocValuesField(LuceneMessageSearchIndex.FIRST_FROM_MAILBOX_NAME_FIELD, new BytesRef((CharSequence)firstFromMailbox)));
                doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.FIRST_TO_MAILBOX_NAME_FIELD, firstToMailbox, Field.Store.YES));
                doc.add((IndexableField)new SortedSetDocValuesField(LuceneMessageSearchIndex.FIRST_TO_MAILBOX_NAME_FIELD, new BytesRef((CharSequence)firstToMailbox)));
                doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.FIRST_CC_MAILBOX_NAME_FIELD, firstCcMailbox, Field.Store.YES));
                doc.add((IndexableField)new SortedSetDocValuesField(LuceneMessageSearchIndex.FIRST_CC_MAILBOX_NAME_FIELD, new BytesRef((CharSequence)firstCcMailbox)));
                doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.FIRST_FROM_MAILBOX_DISPLAY_FIELD, firstFromDisplay, Field.Store.YES));
                doc.add((IndexableField)new StringField(LuceneMessageSearchIndex.FIRST_TO_MAILBOX_DISPLAY_FIELD, firstToDisplay, Field.Store.YES));
            }

            public void body(BodyDescriptor desc, InputStream in) throws MimeException, IOException {
                String mediaType = desc.getMediaType();
                if (LuceneMessageSearchIndex.MEDIA_TYPE_TEXT.equalsIgnoreCase(mediaType) || LuceneMessageSearchIndex.MEDIA_TYPE_MESSAGE.equalsIgnoreCase(mediaType)) {
                    Charset charset;
                    String cset = desc.getCharset();
                    if (cset == null) {
                        cset = LuceneMessageSearchIndex.DEFAULT_ENCODING;
                    }
                    try {
                        charset = Charset.forName(cset);
                    }
                    catch (Exception e) {
                        charset = Charset.forName(LuceneMessageSearchIndex.DEFAULT_ENCODING);
                    }
                    try (BufferedReader bodyReader = new BufferedReader(new InputStreamReader(in, charset));){
                        String line = null;
                        while ((line = bodyReader.readLine()) != null) {
                            doc.add((IndexableField)new TextField(LuceneMessageSearchIndex.BODY_FIELD, line.toUpperCase(Locale.US), Field.Store.NO));
                        }
                    }
                }
            }
        };
        MimeStreamParser parser = new MimeStreamParser(MimeConfig.PERMISSIVE);
        parser.setContentDecoding(true);
        parser.setContentHandler((ContentHandler)handler);
        parser.parse(membership.getFullContent());
        return doc;
    }

    private static boolean hasAttachment(MailboxMessage membership) {
        return MessageAttachmentMetadata.hasNonInlinedAttachment((List)membership.getAttachments());
    }

    private String toSentDateField(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return SENT_DATE_FIELD_YEAR_RESOLUTION;
            }
            case Month: {
                return SENT_DATE_FIELD_MONTH_RESOLUTION;
            }
            case Day: {
                return SENT_DATE_FIELD_DAY_RESOLUTION;
            }
            case Hour: {
                return SENT_DATE_FIELD_HOUR_RESOLUTION;
            }
            case Minute: {
                return SENT_DATE_FIELD_MINUTE_RESOLUTION;
            }
            case Second: {
                return SENT_DATE_FIELD_SECOND_RESOLUTION;
            }
        }
        return SENT_DATE_FIELD_MILLISECOND_RESOLUTION;
    }

    private static Calendar getGMT() {
        return Calendar.getInstance(TimeZone.getTimeZone("GMT"), Locale.US);
    }

    private String toInteralDateField(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return INTERNAL_DATE_FIELD_YEAR_RESOLUTION;
            }
            case Month: {
                return INTERNAL_DATE_FIELD_MONTH_RESOLUTION;
            }
            case Day: {
                return INTERNAL_DATE_FIELD_DAY_RESOLUTION;
            }
            case Hour: {
                return INTERNAL_DATE_FIELD_HOUR_RESOLUTION;
            }
            case Minute: {
                return INTERNAL_DATE_FIELD_MINUTE_RESOLUTION;
            }
            case Second: {
                return INTERNAL_DATE_FIELD_SECOND_RESOLUTION;
            }
        }
        return INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION;
    }

    private String toSaveDateField(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return SAVE_DATE_FIELD_YEAR_RESOLUTION;
            }
            case Month: {
                return SAVE_DATE_FIELD_MONTH_RESOLUTION;
            }
            case Day: {
                return SAVE_DATE_FIELD_DAY_RESOLUTION;
            }
            case Hour: {
                return SAVE_DATE_FIELD_HOUR_RESOLUTION;
            }
            case Minute: {
                return SAVE_DATE_FIELD_MINUTE_RESOLUTION;
            }
            case Second: {
                return SAVE_DATE_FIELD_SECOND_RESOLUTION;
            }
        }
        throw new RuntimeException(String.format("Not support search for the %s date resolution", res));
    }

    private Query createInternalDateQuery(SearchQuery.InternalDateCriterion crit) throws UnsupportedSearchException {
        SearchQuery.DateOperator dop = crit.getOperator();
        SearchQuery.DateResolution res = dop.getDateResultion();
        String field = this.toInteralDateField(res);
        return this.createQuery(field, dop);
    }

    private Query createSaveDateQuery(SearchQuery.SaveDateCriterion crit) throws UnsupportedSearchException {
        SearchQuery.DateOperator dop = crit.getOperator();
        SearchQuery.DateResolution res = dop.getDateResultion();
        String field = this.toSaveDateField(res);
        return this.createQuery(field, dop);
    }

    private Query createSizeQuery(SearchQuery.SizeCriterion crit) throws UnsupportedSearchException {
        SearchQuery.NumericOperator op = crit.getOperator();
        switch (op.getType()) {
            case EQUALS: {
                return LongPoint.newExactQuery((String)SIZE_FIELD, (long)op.getValue());
            }
            case GREATER_THAN: {
                return LongPoint.newRangeQuery((String)SIZE_FIELD, (long)(op.getValue() + 1L), (long)Long.MAX_VALUE);
            }
            case LESS_THAN: {
                return LongPoint.newRangeQuery((String)SIZE_FIELD, (long)Long.MIN_VALUE, (long)(op.getValue() - 1L));
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createTermQuery(String fieldName, String value) {
        if (this.suffixMatch) {
            return new WildcardQuery(new Term(fieldName, "*" + value + "*"));
        }
        return new PrefixQuery(new Term(fieldName, value));
    }

    private Query createHeaderQuery(SearchQuery.HeaderCriterion crit) throws UnsupportedSearchException {
        SearchQuery.HeaderOperator op = crit.getOperator();
        String name = crit.getHeaderName().toUpperCase(Locale.US);
        String fieldName = PREFIX_HEADER_FIELD + name;
        if (op instanceof SearchQuery.ContainsOperator) {
            SearchQuery.ContainsOperator cop = (SearchQuery.ContainsOperator)op;
            return this.createTermQuery(fieldName, cop.getValue().toUpperCase(Locale.US));
        }
        if (op instanceof SearchQuery.ExistsOperator) {
            return new PrefixQuery(new Term(fieldName, ""));
        }
        if (op instanceof SearchQuery.DateOperator) {
            SearchQuery.DateOperator dop = (SearchQuery.DateOperator)op;
            String field = this.toSentDateField(dop.getDateResultion());
            return this.createQuery(field, dop);
        }
        if (op instanceof SearchQuery.AddressOperator) {
            String field = name.toLowerCase(Locale.US);
            return this.createTermQuery(field, ((SearchQuery.AddressOperator)op).getAddress().toUpperCase(Locale.US));
        }
        throw new UnsupportedSearchException();
    }

    private Query createQuery(String field, SearchQuery.DateOperator dop) throws UnsupportedSearchException {
        Date date = dop.getDate();
        SearchQuery.DateResolution res = dop.getDateResultion();
        DateTools.Resolution dRes = this.toResolution(res);
        String value = DateTools.dateToString((Date)date, (DateTools.Resolution)dRes);
        switch (dop.getType()) {
            case ON: {
                return new TermQuery(new Term(field, value));
            }
            case BEFORE: {
                return TermRangeQuery.newStringRange((String)field, (String)DateTools.dateToString((Date)MIN_DATE, (DateTools.Resolution)dRes), (String)value, (boolean)true, (boolean)false);
            }
            case AFTER: {
                return TermRangeQuery.newStringRange((String)field, (String)value, (String)DateTools.dateToString((Date)MAX_DATE, (DateTools.Resolution)dRes), (boolean)false, (boolean)true);
            }
        }
        throw new UnsupportedSearchException();
    }

    private DateTools.Resolution toResolution(SearchQuery.DateResolution res) {
        switch (res) {
            case Year: {
                return DateTools.Resolution.YEAR;
            }
            case Month: {
                return DateTools.Resolution.MONTH;
            }
            case Day: {
                return DateTools.Resolution.DAY;
            }
            case Hour: {
                return DateTools.Resolution.HOUR;
            }
            case Minute: {
                return DateTools.Resolution.MINUTE;
            }
            case Second: {
                return DateTools.Resolution.SECOND;
            }
        }
        return DateTools.Resolution.MILLISECOND;
    }

    private Query createUidQuery(SearchQuery.UidCriterion crit) throws UnsupportedSearchException {
        SearchQuery.UidRange[] ranges = crit.getOperator().getRange();
        if (ranges.length == 1) {
            SearchQuery.UidRange range = ranges[0];
            return LongPoint.newRangeQuery((String)UID_FIELD, (long)range.getLowValue().asLong(), (long)range.getHighValue().asLong());
        }
        BooleanQuery.Builder rangesQuery = new BooleanQuery.Builder();
        for (SearchQuery.UidRange range : ranges) {
            rangesQuery.add(LongPoint.newRangeQuery((String)UID_FIELD, (long)range.getLowValue().asLong(), (long)range.getHighValue().asLong()), BooleanClause.Occur.SHOULD);
        }
        return rangesQuery.build();
    }

    private Query createModSeqQuery(SearchQuery.ModSeqCriterion crit) throws UnsupportedSearchException {
        SearchQuery.NumericOperator op = crit.getOperator();
        switch (op.getType()) {
            case EQUALS: {
                return LongPoint.newRangeQuery((String)MODSEQ_FIELD, (long)op.getValue(), (long)op.getValue());
            }
            case GREATER_THAN: {
                return LongPoint.newRangeQuery((String)MODSEQ_FIELD, (long)op.getValue(), (long)Long.MAX_VALUE);
            }
            case LESS_THAN: {
                return LongPoint.newRangeQuery((String)MODSEQ_FIELD, (long)Long.MIN_VALUE, (long)op.getValue());
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createAttachmentQuery(boolean isSet) throws MailboxException {
        return new TermQuery(new Term(HAS_ATTACHMENT_FIELD, Boolean.toString(isSet)));
    }

    private Query createFlagQuery(String flag, boolean isSet, Query inMailboxes, Collection<MessageUid> recentUids) throws MailboxException {
        Query query;
        block15: {
            BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
            if (isSet) {
                queryBuilder.add((Query)new TermQuery(new Term(FLAGS_FIELD, flag)), BooleanClause.Occur.MUST);
            } else {
                BooleanQuery.Builder bQuery = new BooleanQuery.Builder();
                bQuery.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST);
                bQuery.add((Query)new TermQuery(new Term(FLAGS_FIELD, flag)), BooleanClause.Occur.MUST_NOT);
                queryBuilder.add((Query)bQuery.build(), BooleanClause.Occur.MUST);
            }
            queryBuilder.add(inMailboxes, BooleanClause.Occur.MUST);
            DirectoryReader reader = DirectoryReader.open((IndexWriter)this.writer);
            try {
                ScoreDoc[] sDocs;
                IndexSearcher searcher = new IndexSearcher((IndexReader)reader);
                HashSet<MessageUid> uids = new HashSet<MessageUid>();
                TopFieldDocs docs = searcher.search((Query)queryBuilder.build(), this.maxQueryResults, new Sort(new SortField[]{UID_SORT}));
                for (ScoreDoc sDoc : sDocs = docs.scoreDocs) {
                    MessageUid uid = MessageUid.of((long)searcher.doc(sDoc.doc).getField(UID_FIELD).numericValue().longValue());
                    uids.add(uid);
                }
                if (flag.equalsIgnoreCase("\\RECENT")) {
                    if (isSet) {
                        uids.addAll(recentUids);
                    } else {
                        uids.removeAll(recentUids);
                    }
                }
                List ranges = MessageRange.toRanges(new ArrayList(uids));
                SearchQuery.UidRange[] nRanges = new SearchQuery.UidRange[ranges.size()];
                for (int i = 0; i < ranges.size(); ++i) {
                    MessageRange range = (MessageRange)ranges.get(i);
                    nRanges[i] = new SearchQuery.UidRange(range.getUidFrom(), range.getUidTo());
                }
                query = this.createUidQuery((SearchQuery.UidCriterion)SearchQuery.uid((SearchQuery.UidRange[])nRanges));
                if (reader == null) break block15;
            }
            catch (Throwable throwable) {
                try {
                    if (reader != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new MailboxException("Unable to search mailbox " + String.valueOf(inMailboxes), (Throwable)e);
                }
            }
            reader.close();
        }
        return query;
    }

    private Sort createSort(List<SearchQuery.Sort> sorts) {
        ArrayList<SortField> fields = new ArrayList<SortField>();
        for (SearchQuery.Sort sort : sorts) {
            boolean reverse;
            SortField sortField = this.createSortField(sort, reverse = sort.isReverse());
            if (sortField == null) continue;
            fields.add(sortField);
            if (sortField == SENT_DATE_SORT) {
                fields.add(UID_SORT);
                continue;
            }
            if (sortField != SENT_DATE_SORT_REVERSE) continue;
            fields.add(UID_SORT_REVERSE);
        }
        fields.add(UID_SORT);
        return new Sort((SortField[])fields.toArray(SortField[]::new));
    }

    private SortField createSortField(SearchQuery.Sort s, boolean reverse) {
        switch (s.getSortClause()) {
            case Arrival: {
                if (reverse) {
                    return ARRIVAL_MAILBOX_SORT_REVERSE;
                }
                return ARRIVAL_MAILBOX_SORT;
            }
            case SentDate: {
                if (reverse) {
                    return SENT_DATE_SORT_REVERSE;
                }
                return SENT_DATE_SORT;
            }
            case MailboxCc: {
                if (reverse) {
                    return FIRST_CC_MAILBOX_SORT_REVERSE;
                }
                return FIRST_CC_MAILBOX_SORT;
            }
            case MailboxFrom: {
                if (reverse) {
                    return FIRST_FROM_MAILBOX_SORT_REVERSE;
                }
                return FIRST_FROM_MAILBOX_SORT;
            }
            case Size: {
                if (reverse) {
                    return SIZE_SORT_REVERSE;
                }
                return SIZE_SORT;
            }
            case BaseSubject: {
                if (reverse) {
                    return BASE_SUBJECT_SORT_REVERSE;
                }
                return BASE_SUBJECT_SORT;
            }
            case MailboxTo: {
                if (reverse) {
                    return FIRST_TO_MAILBOX_SORT_REVERSE;
                }
                return FIRST_TO_MAILBOX_SORT;
            }
            case Uid: {
                if (reverse) {
                    return UID_SORT_REVERSE;
                }
                return UID_SORT;
            }
        }
        return null;
    }

    private String toString(Flags.Flag flag) {
        if (Flags.Flag.ANSWERED.equals(flag)) {
            return "\\ANSWERED";
        }
        if (Flags.Flag.DELETED.equals(flag)) {
            return "\\DELETED";
        }
        if (Flags.Flag.DRAFT.equals(flag)) {
            return "\\DRAFT";
        }
        if (Flags.Flag.FLAGGED.equals(flag)) {
            return "\\FLAGGED";
        }
        if (Flags.Flag.RECENT.equals(flag)) {
            return "\\RECENT";
        }
        if (Flags.Flag.SEEN.equals(flag)) {
            return "\\FLAG";
        }
        return flag.toString();
    }

    private Query createTextQuery(SearchQuery.TextCriterion crit) throws UnsupportedSearchException {
        String value = crit.getOperator().getValue().toUpperCase(Locale.US);
        switch (crit.getType()) {
            case BODY: {
                return this.createTermQuery(BODY_FIELD, value);
            }
            case FULL: {
                BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
                queryBuilder.add(this.createTermQuery(BODY_FIELD, value), BooleanClause.Occur.SHOULD);
                queryBuilder.add(this.createTermQuery(HEADERS_FIELD, value), BooleanClause.Occur.SHOULD);
                return queryBuilder.build();
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createAllQuery(SearchQuery.AllCriterion crit) throws UnsupportedSearchException {
        BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
        queryBuilder.add(this.createQuery(MessageRange.all()), BooleanClause.Occur.MUST);
        queryBuilder.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST_NOT);
        return queryBuilder.build();
    }

    private Query createConjunctionQuery(SearchQuery.ConjunctionCriterion crit, Query inMailboxes, Collection<MessageUid> recentUids) throws UnsupportedSearchException, MailboxException {
        List crits = crit.getCriteria();
        BooleanQuery.Builder conQuery = new BooleanQuery.Builder();
        switch (crit.getType()) {
            case AND: {
                for (SearchQuery.Criterion criterion : crits) {
                    conQuery.add(this.createQuery(criterion, inMailboxes, recentUids), BooleanClause.Occur.MUST);
                }
                return conQuery.build();
            }
            case OR: {
                for (SearchQuery.Criterion criterion : crits) {
                    conQuery.add(this.createQuery(criterion, inMailboxes, recentUids), BooleanClause.Occur.SHOULD);
                }
                return conQuery.build();
            }
            case NOR: {
                BooleanQuery.Builder nor = new BooleanQuery.Builder();
                for (SearchQuery.Criterion criterion : crits) {
                    conQuery.add(this.createQuery(criterion, inMailboxes, recentUids), BooleanClause.Occur.SHOULD);
                }
                nor.add(inMailboxes, BooleanClause.Occur.MUST);
                nor.add((Query)conQuery.build(), BooleanClause.Occur.MUST_NOT);
                return nor.build();
            }
        }
        throw new UnsupportedSearchException();
    }

    private Query createQuery(SearchQuery.Criterion criterion, Query inMailboxes, Collection<MessageUid> recentUids) throws MailboxException {
        if (criterion instanceof SearchQuery.InternalDateCriterion) {
            SearchQuery.InternalDateCriterion crit = (SearchQuery.InternalDateCriterion)criterion;
            return this.createInternalDateQuery(crit);
        }
        if (criterion instanceof SearchQuery.SaveDateCriterion) {
            SearchQuery.SaveDateCriterion crit = (SearchQuery.SaveDateCriterion)criterion;
            return this.createSaveDateQuery(crit);
        }
        if (criterion instanceof SearchQuery.SizeCriterion) {
            SearchQuery.SizeCriterion crit = (SearchQuery.SizeCriterion)criterion;
            return this.createSizeQuery(crit);
        }
        if (criterion instanceof SearchQuery.MessageIdCriterion) {
            SearchQuery.MessageIdCriterion crit = (SearchQuery.MessageIdCriterion)criterion;
            return this.createTermQuery(MESSAGE_ID_FIELD, crit.getMessageId().serialize());
        }
        if (criterion instanceof SearchQuery.HeaderCriterion) {
            SearchQuery.HeaderCriterion crit = (SearchQuery.HeaderCriterion)criterion;
            return this.createHeaderQuery(crit);
        }
        if (criterion instanceof SearchQuery.UidCriterion) {
            SearchQuery.UidCriterion crit = (SearchQuery.UidCriterion)criterion;
            return this.createUidQuery(crit);
        }
        if (criterion instanceof SearchQuery.FlagCriterion) {
            SearchQuery.FlagCriterion crit = (SearchQuery.FlagCriterion)criterion;
            return this.createFlagQuery(this.toString(crit.getFlag()), crit.getOperator().isSet(), inMailboxes, recentUids);
        }
        if (criterion instanceof SearchQuery.AttachmentCriterion) {
            SearchQuery.AttachmentCriterion crit = (SearchQuery.AttachmentCriterion)criterion;
            return this.createAttachmentQuery(crit.getOperator().isSet());
        }
        if (criterion instanceof SearchQuery.CustomFlagCriterion) {
            SearchQuery.CustomFlagCriterion crit = (SearchQuery.CustomFlagCriterion)criterion;
            return this.createFlagQuery(crit.getFlag(), crit.getOperator().isSet(), inMailboxes, recentUids);
        }
        if (criterion instanceof SearchQuery.TextCriterion) {
            SearchQuery.TextCriterion crit = (SearchQuery.TextCriterion)criterion;
            return this.createTextQuery(crit);
        }
        if (criterion instanceof SearchQuery.AllCriterion) {
            return this.createAllQuery((SearchQuery.AllCriterion)criterion);
        }
        if (criterion instanceof SearchQuery.ConjunctionCriterion) {
            SearchQuery.ConjunctionCriterion crit = (SearchQuery.ConjunctionCriterion)criterion;
            return this.createConjunctionQuery(crit, inMailboxes, recentUids);
        }
        if (criterion instanceof SearchQuery.ModSeqCriterion) {
            return this.createModSeqQuery((SearchQuery.ModSeqCriterion)criterion);
        }
        if (criterion instanceof SearchQuery.MimeMessageIDCriterion) {
            SearchQuery.MimeMessageIDCriterion mimeMessageIDCriterion = (SearchQuery.MimeMessageIDCriterion)criterion;
            return this.createHeaderQuery(mimeMessageIDCriterion.asHeaderCriterion());
        }
        if (criterion instanceof SearchQuery.SubjectCriterion) {
            SearchQuery.SubjectCriterion subjectCriterion = (SearchQuery.SubjectCriterion)criterion;
            return this.createHeaderQuery(subjectCriterion.asHeaderCriterion());
        }
        if (criterion instanceof SearchQuery.ThreadIdCriterion) {
            SearchQuery.ThreadIdCriterion threadIdCriterion = (SearchQuery.ThreadIdCriterion)criterion;
            return this.createTermQuery(THREAD_ID_FIELD, threadIdCriterion.getThreadId().serialize());
        }
        throw new UnsupportedSearchException();
    }

    public Mono<Void> add(MailboxSession session, org.apache.james.mailbox.model.Mailbox mailbox, MailboxMessage membership) {
        return Mono.fromRunnable((Runnable)Throwing.runnable(() -> {
            Document doc = this.createMessageDocument(session, membership);
            Document flagsDoc = this.createFlagsDocument(membership);
            log.trace("Adding document: uid:'{}' with flags: {}", (Object)doc.get(UID_FIELD), (Object)flagsDoc);
            this.writer.addDocument((Iterable)doc);
            this.writer.addDocument((Iterable)flagsDoc);
        }));
    }

    public Mono<Void> update(MailboxSession session, MailboxId mailboxId, List<UpdatedFlags> updatedFlagsList) {
        return Mono.fromRunnable((Runnable)Throwing.runnable(() -> {
            for (UpdatedFlags updatedFlags : updatedFlagsList) {
                this.update(mailboxId, updatedFlags.getUid(), updatedFlags.getNewFlags());
            }
        }));
    }

    private void update(MailboxId mailboxId, MessageUid uid, Flags f) throws IOException {
        String flagsID = LuceneMessageSearchIndex.createFlagsIdField(mailboxId, uid);
        Term term = new Term(ID_FIELD, flagsID);
        Document doc = this.createFlagsDocument(mailboxId, uid, f);
        log.trace("Updating flags document, mailboxId:{}, message uid: {}, flags:'{}', term: {}, new document: {}", new Object[]{mailboxId, uid, f, term, doc});
        this.writer.updateDocument(term, (Iterable)doc);
    }

    private Document createFlagsDocument(MailboxMessage message) {
        return this.createFlagsDocument(message.getMailboxId(), message.getUid(), message.createFlags());
    }

    private Document createFlagsDocument(MailboxId mailboxId, MessageUid messageUid, Flags flags) {
        Document doc = new Document();
        doc.add((IndexableField)new StringField(ID_FIELD, LuceneMessageSearchIndex.createFlagsIdField(mailboxId, messageUid), Field.Store.YES));
        doc.add((IndexableField)new StringField(MAILBOX_ID_FIELD, mailboxId.serialize(), Field.Store.YES));
        doc.add((IndexableField)new NumericDocValuesField(UID_FIELD, messageUid.asLong()));
        doc.add((IndexableField)new LongPoint(UID_FIELD, new long[]{messageUid.asLong()}));
        doc.add((IndexableField)new StoredField(UID_FIELD, messageUid.asLong()));
        this.indexFlags(doc, flags);
        return doc;
    }

    private static String createFlagsIdField(MailboxId mailboxId, MessageUid messageUid) {
        return "flags-" + mailboxId.serialize() + "-" + messageUid.asLong();
    }

    private void indexFlags(Document doc, Flags f) {
        String[] userFlags;
        Flags.Flag[] flags;
        ArrayList<String> fString = new ArrayList<String>();
        for (Flags.Flag flag : flags = f.getSystemFlags()) {
            fString.add(this.toString(flag));
            doc.add((IndexableField)new StringField(FLAGS_FIELD, this.toString(flag), Field.Store.YES));
        }
        for (String userFlag : userFlags = f.getUserFlags()) {
            doc.add((IndexableField)new StringField(FLAGS_FIELD, userFlag, Field.Store.YES));
        }
        if (flags.length == 0 && userFlags.length == 0) {
            doc.add((IndexableField)new StringField(FLAGS_FIELD, "", Field.Store.NO));
        }
    }

    private Query createQuery(MessageRange range) {
        switch (range.getType()) {
            case ONE: {
                return LongPoint.newRangeQuery((String)UID_FIELD, (long)range.getUidFrom().asLong(), (long)range.getUidTo().asLong());
            }
            case FROM: {
                return LongPoint.newRangeQuery((String)UID_FIELD, (long)range.getUidFrom().asLong(), (long)MessageUid.MAX_VALUE.asLong());
            }
        }
        return LongPoint.newRangeQuery((String)UID_FIELD, (long)MessageUid.MIN_VALUE.asLong(), (long)MessageUid.MAX_VALUE.asLong());
    }

    public Mono<Void> delete(MailboxSession session, MailboxId mailboxId, Collection<MessageUid> expungedUids) {
        return Mono.fromRunnable((Runnable)Throwing.runnable(() -> MessageRange.toRanges((Collection)expungedUids).forEach(Throwing.consumer(messageRange -> this.delete(mailboxId, (MessageRange)messageRange)).sneakyThrow())));
    }

    public Mono<Void> deleteAll(MailboxSession session, MailboxId mailboxId) {
        return Mono.fromRunnable((Runnable)Throwing.runnable(() -> this.delete(mailboxId, MessageRange.all())));
    }

    public void delete(MailboxId mailboxId, MessageRange range) throws IOException {
        BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
        queryBuilder.add((Query)new TermQuery(new Term(MAILBOX_ID_FIELD, mailboxId.serialize())), BooleanClause.Occur.MUST);
        queryBuilder.add(this.createQuery(range), BooleanClause.Occur.MUST);
        this.writer.deleteDocuments(new Query[]{queryBuilder.build()});
    }

    public void commit() throws IOException {
        this.writer.commit();
    }

    public void postReindexing() {
        try {
            this.commit();
        }
        catch (IOException e) {
            throw new RuntimeException("Error while commiting to index", e);
        }
    }

    public Mono<Flags> retrieveIndexedFlags(org.apache.james.mailbox.model.Mailbox mailbox, MessageUid uid) {
        return Mono.fromCallable(() -> this.retrieveFlags(mailbox, uid));
    }

    private Flags retrieveFlags(org.apache.james.mailbox.model.Mailbox mailbox, MessageUid uid) throws IOException {
        try (DirectoryReader reader = DirectoryReader.open((IndexWriter)this.writer);){
            Flags sDocs;
            IndexSearcher searcher = new IndexSearcher((IndexReader)reader);
            Flags retrievedFlags = new Flags();
            BooleanQuery.Builder queryBuilder = new BooleanQuery.Builder();
            queryBuilder.add((Query)new TermQuery(new Term(MAILBOX_ID_FIELD, mailbox.getMailboxId().serialize())), BooleanClause.Occur.MUST);
            queryBuilder.add(this.createQuery(MessageRange.one((MessageUid)uid)), BooleanClause.Occur.MUST);
            queryBuilder.add((Query)new PrefixQuery(new Term(FLAGS_FIELD, "")), BooleanClause.Occur.MUST);
            TopDocs docs = searcher.search((Query)queryBuilder.build(), 100000);
            for (Flags sDoc : sDocs = docs.scoreDocs) {
                Document doc = searcher.doc(sDoc.doc);
                Stream.of(doc.getValues(FLAGS_FIELD)).forEach(flag -> this.fromString((String)flag).ifPresentOrElse(arg_0 -> ((Flags)retrievedFlags).add(arg_0), () -> retrievedFlags.add(flag)));
            }
            Flags flags = retrievedFlags;
            return flags;
        }
    }

    private Optional<Flags.Flag> fromString(String flag) {
        switch (flag) {
            case "\\ANSWERED": {
                return Optional.of(Flags.Flag.ANSWERED);
            }
            case "\\DELETED": {
                return Optional.of(Flags.Flag.DELETED);
            }
            case "\\DRAFT": {
                return Optional.of(Flags.Flag.DRAFT);
            }
            case "\\FLAGGED": {
                return Optional.of(Flags.Flag.FLAGGED);
            }
            case "\\RECENT": {
                return Optional.of(Flags.Flag.RECENT);
            }
            case "\\FLAG": {
                return Optional.of(Flags.Flag.SEEN);
            }
        }
        return Optional.empty();
    }

    static {
        GROUP = new LuceneMessageSearchIndexGroup();
        Calendar cal = Calendar.getInstance();
        cal.set(9999, 11, 31);
        MAX_DATE = cal.getTime();
        cal.set(0, 0, 1);
        MIN_DATE = cal.getTime();
        UID_SORT = new SortField(UID_FIELD, SortField.Type.LONG);
        UID_SORT_REVERSE = new SortField(UID_FIELD, SortField.Type.LONG, true);
        SIZE_SORT = new SortField(SIZE_FIELD, SortField.Type.LONG);
        SIZE_SORT_REVERSE = new SortField(SIZE_FIELD, SortField.Type.LONG, true);
        FIRST_CC_MAILBOX_SORT = new SortedSetSortField(FIRST_CC_MAILBOX_NAME_FIELD, false);
        FIRST_CC_MAILBOX_SORT_REVERSE = new SortedSetSortField(FIRST_CC_MAILBOX_NAME_FIELD, true);
        FIRST_TO_MAILBOX_SORT = new SortedSetSortField(FIRST_TO_MAILBOX_NAME_FIELD, false);
        FIRST_TO_MAILBOX_SORT_REVERSE = new SortedSetSortField(FIRST_TO_MAILBOX_NAME_FIELD, true);
        FIRST_FROM_MAILBOX_SORT = new SortedSetSortField(FIRST_FROM_MAILBOX_NAME_FIELD, false);
        FIRST_FROM_MAILBOX_SORT_REVERSE = new SortedSetSortField(FIRST_FROM_MAILBOX_NAME_FIELD, true);
        ARRIVAL_MAILBOX_SORT = new SortField(INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION, SortField.Type.LONG);
        ARRIVAL_MAILBOX_SORT_REVERSE = new SortField(INTERNAL_DATE_FIELD_MILLISECOND_RESOLUTION, SortField.Type.LONG, true);
        BASE_SUBJECT_SORT = new SortedSetSortField(BASE_SUBJECT_FIELD, false);
        BASE_SUBJECT_SORT_REVERSE = new SortedSetSortField(BASE_SUBJECT_FIELD, true);
        SENT_DATE_SORT = new SortField(SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION, SortField.Type.LONG);
        SENT_DATE_SORT_REVERSE = new SortField(SENT_DATE_SORT_FIELD_MILLISECOND_RESOLUTION, SortField.Type.LONG, true);
    }

    public static class LuceneMessageSearchIndexGroup
    extends org.apache.james.events.Group {
    }
}

